<?php

namespace Tests\Feature;

use App\Models\Incident;
use App\Models\Organization;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class TenantIsolationTest extends TestCase
{
    use RefreshDatabase;

    public function test_org_user_cannot_access_other_org_incidents(): void
    {
        $org1 = Organization::factory()->create();
        $org2 = Organization::factory()->create();

        $user1 = User::factory()->create(['organization_id' => $org1->id]);
        $incident1 = Incident::factory()->create(['organization_id' => $org1->id]);
        $incident2 = Incident::factory()->create(['organization_id' => $org2->id]);

        $response = $this->actingAs($user1, 'sanctum')
            ->getJson('/api/v1/incidents');

        $response->assertStatus(200);
        $response->assertJsonCount(1, 'data');
        $response->assertJsonPath('data.0.id', $incident1->id);
    }

    public function test_citizen_can_only_see_own_incidents(): void
    {
        $org = Organization::factory()->create();
        $citizen1 = User::factory()->create(['organization_id' => $org->id]);
        $citizen1->roles()->attach(\App\Models\Role::factory()->create(['name' => 'CITIZEN']));

        $citizen2 = User::factory()->create(['organization_id' => $org->id]);
        $citizen2->roles()->attach(\App\Models\Role::factory()->create(['name' => 'CITIZEN']));

        $incident1 = Incident::factory()->create([
            'organization_id' => $org->id,
            'reporter_user_id' => $citizen1->id,
        ]);

        $incident2 = Incident::factory()->create([
            'organization_id' => $org->id,
            'reporter_user_id' => $citizen2->id,
        ]);

        $response = $this->actingAs($citizen1, 'sanctum')
            ->getJson('/api/v1/incidents');

        $response->assertStatus(200);
        $response->assertJsonCount(1, 'data');
        $response->assertJsonPath('data.0.id', $incident1->id);
    }

    public function test_super_admin_can_access_all_incidents(): void
    {
        $org1 = Organization::factory()->create();
        $org2 = Organization::factory()->create();

        $superAdmin = User::factory()->create(['organization_id' => null]);
        $superAdmin->roles()->attach(\App\Models\Role::factory()->create(['name' => 'SUPER_ADMIN']));

        $incident1 = Incident::factory()->create(['organization_id' => $org1->id]);
        $incident2 = Incident::factory()->create(['organization_id' => $org2->id]);

        $response = $this->actingAs($superAdmin, 'sanctum')
            ->getJson('/api/v1/incidents');

        $response->assertStatus(200);
        $this->assertGreaterThanOrEqual(2, count($response->json('data')));
    }
}

