<?php

namespace Tests\Feature;

use App\Models\Incident;
use App\Models\Organization;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class StatusTransitionTest extends TestCase
{
    use RefreshDatabase;

    public function test_valid_status_transitions(): void
    {
        $org = Organization::factory()->create();
        $user = User::factory()->create(['organization_id' => $org->id]);
        $user->roles()->attach(\App\Models\Role::factory()->create(['name' => 'DISPATCHER']));

        $incident = Incident::factory()->create([
            'organization_id' => $org->id,
            'status' => 'REPORTED',
        ]);

        // REPORTED -> TRIAGED (valid)
        $response = $this->actingAs($user, 'sanctum')
            ->postJson("/api/v1/incidents/{$incident->id}/status", [
                'status' => 'TRIAGED',
            ]);

        $response->assertStatus(200);
        $this->assertEquals('TRIAGED', $incident->fresh()->status);
    }

    public function test_invalid_status_transitions(): void
    {
        $org = Organization::factory()->create();
        $user = User::factory()->create(['organization_id' => $org->id]);
        $user->roles()->attach(\App\Models\Role::factory()->create(['name' => 'DISPATCHER']));

        $incident = Incident::factory()->create([
            'organization_id' => $org->id,
            'status' => 'REPORTED',
        ]);

        // REPORTED -> RESOLVED (invalid, must go through TRIAGED, ASSIGNED, IN_PROGRESS)
        $response = $this->actingAs($user, 'sanctum')
            ->postJson("/api/v1/incidents/{$incident->id}/status", [
                'status' => 'RESOLVED',
            ]);

        $response->assertStatus(422);
        $this->assertEquals('REPORTED', $incident->fresh()->status);
    }

    public function test_resolved_status_sets_resolved_at(): void
    {
        $org = Organization::factory()->create();
        $user = User::factory()->create(['organization_id' => $org->id]);
        $user->roles()->attach(\App\Models\Role::factory()->create(['name' => 'DISPATCHER']));

        $incident = Incident::factory()->create([
            'organization_id' => $org->id,
            'status' => 'IN_PROGRESS',
        ]);

        $response = $this->actingAs($user, 'sanctum')
            ->postJson("/api/v1/incidents/{$incident->id}/status", [
                'status' => 'RESOLVED',
            ]);

        $response->assertStatus(200);
        $this->assertNotNull($incident->fresh()->resolved_at);
    }
}

