<?php

namespace Tests\Feature;

use App\Models\Incident;
use App\Models\Organization;
use App\Models\Role;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class RbacTest extends TestCase
{
    use RefreshDatabase;

    public function test_only_dispatcher_can_assign_incidents(): void
    {
        $org = Organization::factory()->create();
        $dispatcher = User::factory()->create(['organization_id' => $org->id]);
        $dispatcher->roles()->attach(Role::factory()->create(['name' => 'DISPATCHER']));

        $agent = User::factory()->create(['organization_id' => $org->id]);
        $agent->roles()->attach(Role::factory()->create(['name' => 'FIELD_AGENT']));

        $incident = Incident::factory()->create(['organization_id' => $org->id]);

        // Dispatcher can assign
        $response = $this->actingAs($dispatcher, 'sanctum')
            ->postJson("/api/v1/incidents/{$incident->id}/assign", [
                'assignable_type' => 'user',
                'assignable_id' => $agent->id,
            ]);

        $response->assertStatus(200);

        // Agent cannot assign
        $response = $this->actingAs($agent, 'sanctum')
            ->postJson("/api/v1/incidents/{$incident->id}/assign", [
                'assignable_type' => 'user',
                'assignable_id' => $agent->id,
            ]);

        $response->assertStatus(403);
    }

    public function test_agent_can_only_update_assigned_incidents(): void
    {
        $org = Organization::factory()->create();
        $agent = User::factory()->create(['organization_id' => $org->id]);
        $agent->roles()->attach(Role::factory()->create(['name' => 'FIELD_AGENT']));

        $assignedIncident = Incident::factory()->create([
            'organization_id' => $org->id,
            'assigned_to_type' => 'user',
            'assigned_to_id' => $agent->id,
            'status' => 'ASSIGNED',
        ]);

        $unassignedIncident = Incident::factory()->create([
            'organization_id' => $org->id,
            'status' => 'REPORTED',
        ]);

        // Can update assigned incident
        $response = $this->actingAs($agent, 'sanctum')
            ->patchJson("/api/v1/incidents/{$assignedIncident->id}", [
                'description' => 'Updated description',
            ]);

        $response->assertStatus(200);

        // Cannot update unassigned incident
        $response = $this->actingAs($agent, 'sanctum')
            ->patchJson("/api/v1/incidents/{$unassignedIncident->id}", [
                'description' => 'Updated description',
            ]);

        $response->assertStatus(403);
    }
}

