<?php

namespace Tests\Feature;

use App\Models\Incident;
use App\Models\IncidentStatusEvent;
use App\Models\Organization;
use App\Models\Role;
use App\Models\Team;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class DispatcherAgentFeaturesTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
        
        // Create roles
        $this->dispatcherRole = Role::factory()->create(['name' => 'DISPATCHER']);
        $this->leadAgentRole = Role::factory()->create(['name' => 'LEAD_AGENT']);
        $this->fieldAgentRole = Role::factory()->create(['name' => 'FIELD_AGENT']);
        $this->citizenRole = Role::factory()->create(['name' => 'CITIZEN']);
    }

    public function test_dispatcher_can_view_all_incidents_in_org(): void
    {
        $org = Organization::factory()->create();
        $dispatcher = User::factory()->create(['organization_id' => $org->id]);
        $dispatcher->roles()->attach($this->dispatcherRole);

        $incident1 = Incident::factory()->create(['organization_id' => $org->id]);
        $incident2 = Incident::factory()->create(['organization_id' => $org->id]);

        $response = $this->actingAs($dispatcher, 'sanctum')
            ->getJson('/api/v1/incidents');

        $response->assertStatus(200);
        $response->assertJsonCount(2, 'data');
    }

    public function test_field_agent_can_only_view_assigned_incidents(): void
    {
        $org = Organization::factory()->create();
        $fieldAgent = User::factory()->create(['organization_id' => $org->id]);
        $fieldAgent->roles()->attach($this->fieldAgentRole);

        $assignedIncident = Incident::factory()->create([
            'organization_id' => $org->id,
            'assigned_to_type' => 'user',
            'assigned_to_id' => $fieldAgent->id,
        ]);

        $unassignedIncident = Incident::factory()->create([
            'organization_id' => $org->id,
            'assigned_to_type' => null,
            'assigned_to_id' => null,
        ]);

        $response = $this->actingAs($fieldAgent, 'sanctum')
            ->getJson('/api/v1/incidents');

        $response->assertStatus(200);
        $response->assertJsonCount(1, 'data');
        $response->assertJsonPath('data.0.id', $assignedIncident->id);
    }

    public function test_lead_agent_can_view_team_and_assigned_incidents(): void
    {
        $org = Organization::factory()->create();
        $leadAgent = User::factory()->create(['organization_id' => $org->id]);
        $leadAgent->roles()->attach($this->leadAgentRole);

        $team = Team::create([
            'organization_id' => $org->id,
            'name' => 'Test Team',
            'is_active' => true,
        ]);
        $team->members()->attach($leadAgent);

        $teamIncident = Incident::factory()->create([
            'organization_id' => $org->id,
            'assigned_to_type' => 'team',
            'assigned_to_id' => $team->id,
        ]);

        $assignedIncident = Incident::factory()->create([
            'organization_id' => $org->id,
            'assigned_to_type' => 'user',
            'assigned_to_id' => $leadAgent->id,
        ]);

        $otherIncident = Incident::factory()->create([
            'organization_id' => $org->id,
            'assigned_to_type' => null,
            'assigned_to_id' => null,
        ]);

        $response = $this->actingAs($leadAgent, 'sanctum')
            ->getJson('/api/v1/incidents');

        $response->assertStatus(200);
        $response->assertJsonCount(2, 'data');
    }

    public function test_dispatcher_can_assign_incident_to_agent(): void
    {
        $org = Organization::factory()->create();
        $dispatcher = User::factory()->create(['organization_id' => $org->id]);
        $dispatcher->roles()->attach($this->dispatcherRole);

        $fieldAgent = User::factory()->create(['organization_id' => $org->id]);
        $fieldAgent->roles()->attach($this->fieldAgentRole);

        $incident = Incident::factory()->create([
            'organization_id' => $org->id,
            'status' => 'TRIAGED',
        ]);

        $response = $this->actingAs($dispatcher, 'sanctum')
            ->postJson("/api/v1/incidents/{$incident->id}/assign", [
                'assignable_type' => 'user',
                'assignable_id' => $fieldAgent->id,
                'notes' => 'High priority assignment',
            ]);

        $response->assertStatus(200);
        $response->assertJsonPath('data.status', 'ASSIGNED');
        $response->assertJsonPath('data.assigned_to.id', $fieldAgent->id);
        $response->assertJsonPath('data.assigned_to.type', 'user');

        $this->assertDatabaseHas('incidents', [
            'id' => $incident->id,
            'assigned_to_type' => 'user',
            'assigned_to_id' => $fieldAgent->id,
            'assigned_by_id' => $dispatcher->id,
        ]);
    }

    public function test_dispatcher_can_unassign_incident(): void
    {
        $org = Organization::factory()->create();
        $dispatcher = User::factory()->create(['organization_id' => $org->id]);
        $dispatcher->roles()->attach($this->dispatcherRole);

        $fieldAgent = User::factory()->create(['organization_id' => $org->id]);
        $fieldAgent->roles()->attach($this->fieldAgentRole);

        $incident = Incident::factory()->create([
            'organization_id' => $org->id,
            'assigned_to_type' => 'user',
            'assigned_to_id' => $fieldAgent->id,
            'status' => 'ASSIGNED',
        ]);

        $response = $this->actingAs($dispatcher, 'sanctum')
            ->postJson("/api/v1/incidents/{$incident->id}/unassign", [
                'note' => 'Reassigning to different agent',
            ]);

        $response->assertStatus(200);
        $response->assertJsonPath('data.assigned_to_id', null);

        $this->assertDatabaseHas('incidents', [
            'id' => $incident->id,
            'assigned_to_type' => null,
            'assigned_to_id' => null,
        ]);
    }

    public function test_field_agent_cannot_assign_incidents(): void
    {
        $org = Organization::factory()->create();
        $fieldAgent = User::factory()->create(['organization_id' => $org->id]);
        $fieldAgent->roles()->attach($this->fieldAgentRole);

        $otherAgent = User::factory()->create(['organization_id' => $org->id]);
        $otherAgent->roles()->attach($this->fieldAgentRole);

        $incident = Incident::factory()->create([
            'organization_id' => $org->id,
            'status' => 'TRIAGED',
        ]);

        $response = $this->actingAs($fieldAgent, 'sanctum')
            ->postJson("/api/v1/incidents/{$incident->id}/assign", [
                'assignable_type' => 'user',
                'assignable_id' => $otherAgent->id,
            ]);

        $response->assertStatus(403);
    }

    public function test_field_agent_can_update_assigned_incident_status(): void
    {
        $org = Organization::factory()->create();
        $fieldAgent = User::factory()->create(['organization_id' => $org->id]);
        $fieldAgent->roles()->attach($this->fieldAgentRole);

        $incident = Incident::factory()->create([
            'organization_id' => $org->id,
            'assigned_to_type' => 'user',
            'assigned_to_id' => $fieldAgent->id,
            'status' => 'ASSIGNED',
        ]);

        $response = $this->actingAs($fieldAgent, 'sanctum')
            ->postJson("/api/v1/incidents/{$incident->id}/status", [
                'status' => 'IN_PROGRESS',
                'notes' => 'Starting work on site',
            ]);

        $response->assertStatus(200);
        // Verify database was updated
        $this->assertDatabaseHas('incidents', [
            'id' => $incident->id,
            'status' => 'IN_PROGRESS',
        ]);
        // Check response - status should be in data.status
        $response->assertJsonPath('data.status', 'IN_PROGRESS');
    }

    public function test_field_agent_cannot_update_unassigned_incident(): void
    {
        $org = Organization::factory()->create();
        $fieldAgent = User::factory()->create(['organization_id' => $org->id]);
        $fieldAgent->roles()->attach($this->fieldAgentRole);

        $incident = Incident::factory()->create([
            'organization_id' => $org->id,
            'assigned_to_type' => null,
            'assigned_to_id' => null,
            'status' => 'TRIAGED',
        ]);

        $response = $this->actingAs($fieldAgent, 'sanctum')
            ->postJson("/api/v1/incidents/{$incident->id}/status", [
                'status' => 'IN_PROGRESS',
            ]);

        $response->assertStatus(403);
    }

    public function test_dispatcher_can_add_internal_note(): void
    {
        $org = Organization::factory()->create();
        $dispatcher = User::factory()->create(['organization_id' => $org->id]);
        $dispatcher->roles()->attach($this->dispatcherRole);

        $incident = Incident::factory()->create(['organization_id' => $org->id]);

        $response = $this->actingAs($dispatcher, 'sanctum')
            ->postJson("/api/v1/incidents/{$incident->id}/timeline", [
                'event_type' => 'INTERNAL_NOTE',
                'notes' => 'Waiting for equipment delivery',
                'visibility' => 'INTERNAL_ONLY',
            ]);

        $response->assertStatus(201);
        $response->assertJsonPath('event_type', 'INTERNAL_NOTE');
        $response->assertJsonPath('visibility', 'INTERNAL_ONLY');

        $this->assertDatabaseHas('incident_status_events', [
            'incident_id' => $incident->id,
            'event_type' => 'INTERNAL_NOTE',
            'visibility' => 'INTERNAL_ONLY',
            'user_id' => $dispatcher->id,
        ]);
    }

    public function test_citizen_cannot_see_internal_notes_in_timeline(): void
    {
        $org = Organization::factory()->create();
        $dispatcher = User::factory()->create(['organization_id' => $org->id]);
        $dispatcher->roles()->attach($this->dispatcherRole);

        $citizen = User::factory()->create(['organization_id' => $org->id]);
        $citizen->roles()->attach($this->citizenRole);

        $incident = Incident::factory()->create([
            'organization_id' => $org->id,
            'reporter_user_id' => $citizen->id,
        ]);

        // Dispatcher adds internal note
        IncidentStatusEvent::create([
            'incident_id' => $incident->id,
            'user_id' => $dispatcher->id,
            'event_type' => 'INTERNAL_NOTE',
            'to_status' => $incident->status,
            'visibility' => 'INTERNAL_ONLY',
            'notes' => 'Internal note',
        ]);

        // Public status change
        IncidentStatusEvent::create([
            'incident_id' => $incident->id,
            'user_id' => $dispatcher->id,
            'event_type' => 'STATUS_CHANGE',
            'visibility' => 'PUBLIC_TO_CITIZEN',
            'from_status' => 'REPORTED',
            'to_status' => 'TRIAGED',
        ]);

        $response = $this->actingAs($citizen, 'sanctum')
            ->getJson("/api/v1/incidents/{$incident->id}/timeline");

        $response->assertStatus(200);
        $timeline = $response->json('timeline');
        
        // Citizen should only see public events
        $this->assertCount(1, $timeline);
        $this->assertEquals('STATUS_CHANGE', $timeline[0]['event_type']);
    }

    public function test_tenant_isolation_dispatcher_cannot_access_other_org(): void
    {
        $org1 = Organization::factory()->create();
        $org2 = Organization::factory()->create();

        $dispatcher = User::factory()->create(['organization_id' => $org1->id]);
        $dispatcher->roles()->attach($this->dispatcherRole);

        $incident = Incident::factory()->create(['organization_id' => $org2->id]);

        $response = $this->actingAs($dispatcher, 'sanctum')
            ->getJson("/api/v1/incidents/{$incident->id}");

        $response->assertStatus(403);
    }

    public function test_filter_unassigned_incidents(): void
    {
        $org = Organization::factory()->create();
        $dispatcher = User::factory()->create(['organization_id' => $org->id]);
        $dispatcher->roles()->attach($this->dispatcherRole);

        $assignedIncident = Incident::factory()->create([
            'organization_id' => $org->id,
            'assigned_to_type' => 'user',
            'assigned_to_id' => User::factory()->create(['organization_id' => $org->id])->id,
        ]);

        $unassignedIncident = Incident::factory()->create([
            'organization_id' => $org->id,
            'assigned_to_type' => null,
            'assigned_to_id' => null,
        ]);

        $response = $this->actingAs($dispatcher, 'sanctum')
            ->getJson('/api/v1/incidents?unassigned=true');

        $response->assertStatus(200);
        $response->assertJsonCount(1, 'data');
        $response->assertJsonPath('data.0.id', $unassignedIncident->id);
    }

    public function test_filter_mine_incidents(): void
    {
        $org = Organization::factory()->create();
        $fieldAgent = User::factory()->create(['organization_id' => $org->id]);
        $fieldAgent->roles()->attach($this->fieldAgentRole);

        $myIncident = Incident::factory()->create([
            'organization_id' => $org->id,
            'assigned_to_type' => 'user',
            'assigned_to_id' => $fieldAgent->id,
        ]);

        $otherIncident = Incident::factory()->create([
            'organization_id' => $org->id,
            'assigned_to_type' => 'user',
            'assigned_to_id' => User::factory()->create(['organization_id' => $org->id])->id,
        ]);

        $response = $this->actingAs($fieldAgent, 'sanctum')
            ->getJson('/api/v1/incidents?mine=true');

        $response->assertStatus(200);
        $response->assertJsonCount(1, 'data');
        $response->assertJsonPath('data.0.id', $myIncident->id);
    }

    public function test_dispatcher_can_triage_incident(): void
    {
        $org = Organization::factory()->create();
        $dispatcher = User::factory()->create(['organization_id' => $org->id]);
        $dispatcher->roles()->attach($this->dispatcherRole);

        $incident = Incident::factory()->create([
            'organization_id' => $org->id,
            'status' => 'REPORTED',
        ]);

        $response = $this->actingAs($dispatcher, 'sanctum')
            ->postJson("/api/v1/incidents/{$incident->id}/status", [
                'status' => 'TRIAGED',
                'notes' => 'Verified and triaged',
            ]);

        $response->assertStatus(200);
        // Verify database was updated first
        $this->assertDatabaseHas('incidents', [
            'id' => $incident->id,
            'status' => 'TRIAGED',
            'triaged_by_id' => $dispatcher->id,
        ]);
        // Then check response
        $response->assertJsonPath('data.status', 'TRIAGED');
    }
}
