<?php

namespace Tests\Feature;

use App\Models\DeviceToken;
use App\Models\Incident;
use App\Models\IncidentMedia;
use App\Models\Organization;
use App\Models\Role;
use App\Models\ServiceArea;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use Tests\TestCase;

class CitizenAgentMvpTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
        Storage::fake('public');
    }

    /**
     * Test: Citizen can create incident and backend routes it to an org
     */
    public function test_citizen_can_create_incident_and_backend_routes_to_org(): void
    {
        $org = Organization::factory()->create(['name' => 'Test Organization']);
        
        ServiceArea::factory()->create([
            'organization_id' => $org->id,
            'service_type' => 'electricity',
            'min_latitude' => 5.25,
            'max_latitude' => 5.35,
            'min_longitude' => -4.05,
            'max_longitude' => -3.95,
        ]);

        $citizen = User::factory()->create();
        $citizenRole = Role::factory()->create(['name' => 'CITIZEN']);
        $citizen->roles()->attach($citizenRole);

        $response = $this->actingAs($citizen, 'sanctum')
            ->postJson('/api/v1/incidents', [
                'service_type' => 'electricity',
                'title' => 'Power Outage',
                'description' => 'No power in my area',
                'latitude' => 5.30,
                'longitude' => -4.00,
                'address_text' => 'Test Address',
            ]);

        $response->assertStatus(201);
        $response->assertJsonPath('data.organization_id', $org->id);
        $response->assertJsonPath('data.organization.name', 'Test Organization');
        
        $this->assertDatabaseHas('incidents', [
            'reporter_user_id' => $citizen->id,
            'organization_id' => $org->id,
            'status' => 'REPORTED',
        ]);
    }

    /**
     * Test: Citizen cannot access incidents they didn't create
     */
    public function test_citizen_cannot_access_incidents_they_did_not_create(): void
    {
        $org = Organization::factory()->create();
        $citizen1 = User::factory()->create();
        $citizen2 = User::factory()->create();
        
        $citizenRole = Role::factory()->create(['name' => 'CITIZEN']);
        $citizen1->roles()->attach($citizenRole);
        $citizen2->roles()->attach($citizenRole);

        $incident = Incident::factory()->create([
            'organization_id' => $org->id,
            'reporter_user_id' => $citizen1->id,
        ]);

        // Citizen 2 tries to access Citizen 1's incident
        $response = $this->actingAs($citizen2, 'sanctum')
            ->getJson("/api/v1/incidents/{$incident->id}");

        $response->assertStatus(403);
    }

    /**
     * Test: Citizen can only see their own incidents in list
     */
    public function test_citizen_can_only_see_own_incidents_in_list(): void
    {
        $org = Organization::factory()->create();
        $citizen1 = User::factory()->create();
        $citizen2 = User::factory()->create();
        
        $citizenRole = Role::factory()->create(['name' => 'CITIZEN']);
        $citizen1->roles()->attach($citizenRole);
        $citizen2->roles()->attach($citizenRole);

        $incident1 = Incident::factory()->create([
            'organization_id' => $org->id,
            'reporter_user_id' => $citizen1->id,
        ]);

        $incident2 = Incident::factory()->create([
            'organization_id' => $org->id,
            'reporter_user_id' => $citizen2->id,
        ]);

        $response = $this->actingAs($citizen1, 'sanctum')
            ->getJson('/api/v1/incidents');

        $response->assertStatus(200);
        $data = $response->json('data');
        
        // Should only see incident1, not incident2
        $incidentIds = collect($data)->pluck('id')->toArray();
        $this->assertContains($incident1->id, $incidentIds);
        $this->assertNotContains($incident2->id, $incidentIds);
    }

    /**
     * Test: Media upload accepts image and audio
     */
    public function test_media_upload_accepts_image_and_audio(): void
    {
        $org = Organization::factory()->create();
        ServiceArea::factory()->create([
            'organization_id' => $org->id,
            'service_type' => 'electricity',
            'min_latitude' => 5.25,
            'max_latitude' => 5.35,
            'min_longitude' => -4.05,
            'max_longitude' => -3.95,
        ]);

        $citizen = User::factory()->create();
        $citizenRole = Role::factory()->create(['name' => 'CITIZEN']);
        $citizen->roles()->attach($citizenRole);

        $image = UploadedFile::fake()->image('test.jpg', 100, 100);
        $audio = UploadedFile::fake()->create('voice.m4a', 100, 'audio/m4a');

        $response = $this->actingAs($citizen, 'sanctum')
            ->post('/api/v1/incidents', [
                'service_type' => 'electricity',
                'title' => 'Test with Media',
                'description' => 'Test description',
                'latitude' => 5.30,
                'longitude' => -4.00,
                'media' => [$image, $audio],
            ]);

        $response->assertStatus(201);
        
        $incident = Incident::latest()->first();
        $this->assertEquals(2, $incident->media()->count());
        
        $mediaTypes = $incident->media()->pluck('media_type')->toArray();
        $this->assertContains('image', $mediaTypes);
        $this->assertContains('audio', $mediaTypes);
    }

    /**
     * Test: Device token upsert works
     */
    public function test_device_token_upsert_works(): void
    {
        $user = User::factory()->create();
        $agentRole = Role::factory()->create(['name' => 'FIELD_AGENT']);
        $user->roles()->attach($agentRole);

        // First registration
        $response = $this->actingAs($user, 'sanctum')
            ->postJson('/api/v1/device-tokens', [
                'token' => 'test_fcm_token_123',
                'device_type' => 'android',
                'device_id' => 'device_123',
            ]);

        $response->assertStatus(201);
        $this->assertDatabaseHas('device_tokens', [
            'user_id' => $user->id,
            'token' => 'test_fcm_token_123',
            'device_type' => 'android',
            'device_id' => 'device_123',
            'is_active' => true,
        ]);

        // Update with new token (upsert)
        $response = $this->actingAs($user, 'sanctum')
            ->postJson('/api/v1/device-tokens', [
                'token' => 'test_fcm_token_456',
                'device_type' => 'android',
                'device_id' => 'device_123',
            ]);

        $response->assertStatus(201);
        
        // Should update existing, not create duplicate
        $this->assertDatabaseCount('device_tokens', 1);
        $this->assertDatabaseHas('device_tokens', [
            'user_id' => $user->id,
            'token' => 'test_fcm_token_456',
            'device_id' => 'device_123',
        ]);
    }

    /**
     * Test: Separate media upload endpoint works
     */
    public function test_separate_media_upload_endpoint_works(): void
    {
        $org = Organization::factory()->create();
        $agent = User::factory()->create();
        $agentRole = Role::factory()->create(['name' => 'FIELD_AGENT']);
        $agent->roles()->attach($agentRole);

        $incident = Incident::factory()->create([
            'organization_id' => $org->id,
            'assigned_to_id' => $agent->id,
            'assigned_to_type' => 'user',
        ]);

        $image = UploadedFile::fake()->image('additional.jpg', 100, 100);
        $audio = UploadedFile::fake()->create('voice_note.m4a', 100, 'audio/m4a');

        $response = $this->actingAs($agent, 'sanctum')
            ->post("/api/v1/incidents/{$incident->id}/media", [
                'media' => [$image, $audio],
            ]);

        $response->assertStatus(201);
        $response->assertJsonStructure([
            'message',
            'media' => [
                '*' => ['id', 'file_name', 'url', 'media_type']
            ]
        ]);

        $this->assertEquals(2, $incident->fresh()->media()->count());
    }
}

