<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;

/**
 * Master seeder that deletes all incident-related data and recreates it
 * 
 * This seeder ensures data accuracy by:
 * 1. Deleting all incident-related data (comments, notifications, assignments, status events, media, incidents)
 * 2. Recreating everything in the correct order:
 *    - Incidents (with media and status events)
 *    - Assignments (created during incident seeding)
 *    - Comments (with media)
 *    - Notifications (from incidents, assignments, status events, and comments)
 * 
 * Usage:
 * php artisan db:seed --class=RefreshAllIncidentDataSeeder
 */
class RefreshAllIncidentDataSeeder extends Seeder
{
    public function run(): void
    {
        $this->command->info('🔄 Starting comprehensive incident data refresh...');
        $this->command->newLine();
        
        // Step 1: Delete all incident-related data
        $this->command->info('📋 Step 1: Deleting all incident-related data...');
        $this->call(DeleteAllIncidentDataSeeder::class);
        $this->command->newLine();
        
        // Step 2: Recreate incidents (includes media and status events)
        $this->command->info('📋 Step 2: Creating incidents with media and status events...');
        $this->call(IncidentSeeder::class);
        $this->command->newLine();
        
        // Step 3: Create comments (includes comment media and comment notifications)
        $this->command->info('📋 Step 3: Creating comments with media and notifications...');
        $this->call(RecreateCommentsSeeder::class);
        $this->command->newLine();
        
        // Step 4: Create notifications (from incidents, assignments, and status events)
        // Note: Comment notifications are already created by RecreateCommentsSeeder
        $this->command->info('📋 Step 4: Creating notifications from incidents, assignments, and status events...');
        $this->call(RecreateNotificationsSeeder::class);
        $this->command->newLine();
        
        $this->command->info('✅ All incident data has been refreshed successfully!');
        $this->command->info('   - Incidents created');
        $this->command->info('   - Assignments created');
        $this->command->info('   - Comments created');
        $this->command->info('   - Notifications created');
    }
}

