<?php

namespace Database\Seeders;

use App\Models\AuditLog;
use App\Models\Incident;
use App\Models\Assignment;
use App\Models\IncidentStatusEvent;
use App\Models\User;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;

class RecreateNotificationsSeeder extends Seeder
{
    public function run(): void
    {
        $this->command->info('Deleting existing notifications...');
        
        // Delete all notification-related audit logs (except comment_added, which is handled by RecreateCommentsSeeder)
        $deleted = DB::table('audit_logs')
            ->where('entity_type', 'App\Models\Incident')
            ->whereIn('action', ['created', 'assigned', 'status_changed'])
            ->delete();
        
        $this->command->info("Deleted {$deleted} notification entries from audit_logs.");
        
        $this->command->info('Recreating notifications from database data...');
        
        $notificationCount = 0;
        
        // 1. Create "created" notifications for all incidents
        $incidents = Incident::with('organization', 'reporter')->get();
        $this->command->info("Processing {$incidents->count()} incidents for 'created' notifications...");
        
        foreach ($incidents as $incident) {
            // Use reporter if available, otherwise use a random user from the organization
            $userId = $incident->reporter_user_id;
            if (!$userId) {
                $orgUser = User::where('organization_id', $incident->organization_id)
                    ->whereHas('roles', fn($q) => $q->whereIn('name', ['CITIZEN', 'FIELD_AGENT', 'LEAD_AGENT', 'DISPATCHER']))
                    ->first();
                $userId = $orgUser?->id;
            }
            
            if (!$userId) {
                continue; // Skip if no user found
            }
            
            AuditLog::create([
                'organization_id' => $incident->organization_id,
                'user_id' => $userId,
                'action' => 'created',
                'entity_type' => 'App\Models\Incident',
                'entity_id' => $incident->id,
                'before' => null,
                'after' => [
                    'id' => $incident->id,
                    'title' => $incident->title,
                    'status' => 'REPORTED',
                ],
                'ip_address' => $this->randomIpAddress(),
                'user_agent' => $this->randomUserAgent(),
                'created_at' => $incident->created_at,
                'updated_at' => $incident->created_at,
            ]);
            $notificationCount++;
        }
        
        $this->command->info("Created {$notificationCount} 'created' notifications.");
        
        // 2. Create "assigned" notifications from assignments table
        $assignments = Assignment::with(['incident.organization'])
            ->whereNotNull('assigned_at')
            ->whereNull('unassigned_at')
            ->get();
        
        $this->command->info("Processing {$assignments->count()} assignments for 'assigned' notifications...");
        
        $assignedCount = 0;
        foreach ($assignments as $assignment) {
            $incident = $assignment->incident;
            if (!$incident) {
                continue;
            }
            
            // Get the assigner (assigned_by) or use a dispatcher from the organization
            $assignerId = $assignment->assigned_by;
            if (!$assignerId) {
                $assigner = User::where('organization_id', $incident->organization_id)
                    ->whereHas('roles', fn($q) => $q->whereIn('name', ['DISPATCHER', 'ORG_ADMIN']))
                    ->first();
                $assignerId = $assigner?->id;
            }
            
            if (!$assignerId) {
                continue;
            }
            
            // Get the assigned user/team ID
            $assignedId = $assignment->assignable_id;
            $assignedType = $assignment->assignable_type;
            
            // Determine previous status (likely TRIAGED before assignment)
            $previousStatus = 'TRIAGED';
            $firstStatusEvent = IncidentStatusEvent::where('incident_id', $incident->id)
                ->whereNotNull('to_status')
                ->orderBy('created_at')
                ->first();
            
            if ($firstStatusEvent && $firstStatusEvent->to_status) {
                $previousStatus = $firstStatusEvent->to_status;
            }
            
            AuditLog::create([
                'organization_id' => $incident->organization_id,
                'user_id' => $assignerId,
                'action' => 'assigned',
                'entity_type' => 'App\Models\Incident',
                'entity_id' => $incident->id,
                'before' => [
                    'assigned_to_id' => null,
                    'status' => $previousStatus,
                ],
                'after' => [
                    'assigned_to_id' => $assignedId,
                    'assigned_to_type' => $assignedType === 'App\Models\User' ? 'user' : 'team',
                    'status' => 'ASSIGNED',
                ],
                'ip_address' => $this->randomIpAddress(),
                'user_agent' => $this->randomUserAgent(),
                'created_at' => $assignment->assigned_at,
                'updated_at' => $assignment->assigned_at,
            ]);
            $assignedCount++;
        }
        
        $this->command->info("Created {$assignedCount} 'assigned' notifications.");
        $notificationCount += $assignedCount;
        
        // 3. Create "status_changed" notifications from incident_status_events
        $statusEvents = IncidentStatusEvent::with(['incident.organization', 'user'])
            ->whereNotNull('to_status')
            ->orderBy('incident_id')
            ->orderBy('created_at')
            ->get();
        
        $this->command->info("Processing {$statusEvents->count()} status events for 'status_changed' notifications...");
        
        $statusChangedCount = 0;
        $previousStatusByIncident = [];
        
        foreach ($statusEvents as $event) {
            $incident = $event->incident;
            if (!$incident) {
                continue;
            }
            
            // Track previous status per incident
            if (!isset($previousStatusByIncident[$incident->id])) {
                $previousStatusByIncident[$incident->id] = 'REPORTED';
            }
            
            $fromStatus = $event->from_status ?? $previousStatusByIncident[$incident->id];
            $toStatus = $event->to_status;
            
            // Use the user from the event, or find a user from the organization
            $userId = $event->user_id;
            if (!$userId) {
                $orgUser = User::where('organization_id', $incident->organization_id)
                    ->whereHas('roles', fn($q) => $q->whereIn('name', ['FIELD_AGENT', 'LEAD_AGENT', 'DISPATCHER', 'ORG_ADMIN']))
                    ->first();
                $userId = $orgUser?->id;
            }
            
            if (!$userId) {
                // Update previous status even if we skip creating the notification
                $previousStatusByIncident[$incident->id] = $toStatus;
                continue;
            }
            
            AuditLog::create([
                'organization_id' => $incident->organization_id,
                'user_id' => $userId,
                'action' => 'status_changed',
                'entity_type' => 'App\Models\Incident',
                'entity_id' => $incident->id,
                'before' => [
                    'status' => $fromStatus,
                ],
                'after' => [
                    'status' => $toStatus,
                ],
                'ip_address' => $this->randomIpAddress(),
                'user_agent' => $this->randomUserAgent(),
                'created_at' => $event->created_at,
                'updated_at' => $event->created_at,
            ]);
            
            $previousStatusByIncident[$incident->id] = $toStatus;
            $statusChangedCount++;
        }
        
        $this->command->info("Created {$statusChangedCount} 'status_changed' notifications.");
        $notificationCount += $statusChangedCount;
        
        $this->command->info("✅ Successfully recreated {$notificationCount} notification entries in audit_logs.");
    }
    
    private function randomIpAddress(): string
    {
        return rand(1, 255) . '.' . rand(1, 255) . '.' . rand(1, 255) . '.' . rand(1, 255);
    }
    
    private function randomUserAgent(): string
    {
        $userAgents = [
            'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
            'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_7) AppleWebKit/537.36',
            'Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36',
            'Mozilla/5.0 (iPhone; CPU iPhone OS 14_0 like Mac OS X) AppleWebKit/605.1.15',
            'Mozilla/5.0 (Android 11; Mobile; rv:68.0) Gecko/68.0',
        ];
        return $userAgents[array_rand($userAgents)];
    }
}

