<?php

namespace Database\Seeders;

use App\Models\AuditLog;
use App\Models\Comment;
use App\Models\CommentMedia;
use App\Models\Incident;
use App\Models\User;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;

class RecreateCommentsSeeder extends Seeder
{
    public function run(): void
    {
        $this->command->info('Deleting existing comments...');
        
        // Delete all comment media first
        DB::table('comment_media')->delete();
        
        // Delete all comments
        $deleted = DB::table('comments')->delete();
        
        $this->command->info("Deleted {$deleted} comments and all related media.");
        
        $this->command->info('Recreating comments from current incidents...');
        
        // Get users who can comment
        $users = User::whereHas('roles', fn($q) => $q->whereIn('name', ['FIELD_AGENT', 'LEAD_AGENT', 'DISPATCHER']))
            ->get();
        
        if ($users->isEmpty()) {
            $this->command->info('No users found. Skipping comment seeding.');
            return;
        }
        
        // Get all incidents
        $incidents = Incident::with('organization')->get();
        
        if ($incidents->isEmpty()) {
            $this->command->info('No incidents found. Skipping comment seeding.');
            return;
        }
        
        $this->command->info("Processing {$incidents->count()} incidents for comments...");
        
        $commentBodies = [
            'Inspection sur site terminée. Problème confirmé.',
            'En attente de l\'arrivée des pièces. Livraison estimée: 2 jours.',
            'Travaux en cours. Mise à jour une fois terminé.',
            'Photos supplémentaires jointes pour référence.',
            'Équipe envoyée sur les lieux. ETA 30 minutes.',
            'Problème résolu. Veuillez vérifier.',
            'Ressources supplémentaires nécessaires. Escalade au superviseur.',
            'Conditions météorologiques retardant les travaux. Reprise demain.',
            'Client contacté. Rendez-vous programmé.',
            'Suivi requis. Client non disponible.',
            'Dysfonctionnement de l\'équipement. Demande d\'équipe de secours.',
            'Zone sécurisée. Poursuite des réparations.',
            'Documentation mise à jour. Toutes les photos jointes.',
            'Mise à jour du statut: 50% terminé.',
            'Inspection finale en attente.',
            'Travaux commencés. Progression normale.',
            'Problème identifié. Solution en cours.',
            'Mise à jour: travaux presque terminés.',
            'Client informé du progrès.',
            'Rapport final en préparation.',
        ];
        
        $imageFileNames = [
            'inspection-photo-1.jpg',
            'work-progress-1.jpg',
            'issue-detail-1.jpg',
            'before-repair-1.jpg',
            'after-repair-1.jpg',
            'equipment-photo-1.jpg',
            'location-marker-1.jpg',
            'damage-assessment-1.jpg',
            'completed-work-1.jpg',
            'safety-check-1.jpg',
        ];
        
        $commentCount = 0;
        $notificationCount = 0;
        
        foreach ($incidents as $incident) {
            // Only add comments to incidents that are assigned, in progress, or resolved
            if (!in_array($incident->status, ['ASSIGNED', 'IN_PROGRESS', 'RESOLVED'])) {
                continue;
            }
            
            // 60% chance of having comments
            if (rand(1, 10) <= 6) {
                $numComments = rand(1, 5); // 1-5 comments per incident
                
                for ($i = 0; $i < $numComments; $i++) {
                    $user = $users->random();
                    $createdAt = now()->subDays(rand(0, 30))->subHours(rand(0, 23));
                    
                    $comment = Comment::create([
                        'commentable_type' => Incident::class,
                        'commentable_id' => $incident->id,
                        'user_id' => $user->id,
                        'body' => $commentBodies[array_rand($commentBodies)],
                        'created_at' => $createdAt,
                        'updated_at' => $createdAt,
                    ]);
                    
                    $commentCount++;
                    
                    // 50% chance of having images in comment
                    if (rand(1, 10) <= 5) {
                        $imageCount = rand(1, 3); // 1-3 images per comment
                        for ($img = 0; $img < $imageCount; $img++) {
                            $fileName = $imageFileNames[array_rand($imageFileNames)];
                            // Use picsum.photos for working placeholder images
                            $imageUrl = 'https://picsum.photos/800/600?random=' . ($comment->id * 10 + $img + 1);
                            CommentMedia::create([
                                'comment_id' => $comment->id,
                                'file_path' => $imageUrl, // Store full URL for external images
                                'file_name' => $fileName,
                                'mime_type' => 'image/jpeg',
                                'file_size' => rand(500000, 2000000), // 500KB - 2MB
                                'media_type' => 'image',
                                'order' => $img,
                            ]);
                        }
                    }
                    
                    // Create notification (audit log entry) for comment
                    AuditLog::create([
                        'organization_id' => $incident->organization_id,
                        'user_id' => $user->id,
                        'action' => 'comment_added',
                        'entity_type' => Incident::class,
                        'entity_id' => $incident->id,
                        'before' => null,
                        'after' => [
                            'comment_id' => $comment->id,
                            'comment_body' => substr($comment->body, 0, 100), // First 100 chars
                        ],
                        'ip_address' => $this->randomIpAddress(),
                        'user_agent' => $this->randomUserAgent(),
                        'created_at' => $createdAt,
                        'updated_at' => $createdAt,
                    ]);
                    $notificationCount++;
                }
            }
        }
        
        $this->command->info("✅ Created {$commentCount} comments.");
        $this->command->info("✅ Created {$notificationCount} comment notifications in audit_logs.");
    }
    
    private function randomIpAddress(): string
    {
        return rand(1, 255) . '.' . rand(1, 255) . '.' . rand(1, 255) . '.' . rand(1, 255);
    }
    
    private function randomUserAgent(): string
    {
        $userAgents = [
            'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
            'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_7) AppleWebKit/537.36',
            'Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36',
            'Mozilla/5.0 (iPhone; CPU iPhone OS 14_0 like Mac OS X) AppleWebKit/605.1.15',
            'Mozilla/5.0 (Android 11; Mobile; rv:68.0) Gecko/68.0',
        ];
        return $userAgents[array_rand($userAgents)];
    }
}

