<?php

namespace Database\Seeders;

use App\Models\AuditLog;
use App\Models\Incident;
use App\Models\User;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;

class NotificationSeeder extends Seeder
{
    public function run(): void
    {
        // Get users with agent/dispatcher roles
        $agents = User::whereHas('roles', fn($q) => $q->whereIn('name', ['FIELD_AGENT', 'LEAD_AGENT', 'DISPATCHER']))
            ->get();

        if ($agents->isEmpty()) {
            $this->command->info('No agents found. Skipping notification seeding.');
            return;
        }

        // Get incidents for notifications
        $incidents = Incident::with('organization')
            ->whereIn('status', ['ASSIGNED', 'IN_PROGRESS', 'RESOLVED', 'TRIAGED'])
            ->limit(200)
            ->get();

        if ($incidents->isEmpty()) {
            $this->command->info('No incidents found. Skipping notification seeding.');
            return;
        }

        $this->command->info("Creating notifications for {$incidents->count()} incidents...");

        $notificationCount = 0;

        foreach ($incidents as $incident) {
            $organizationId = $incident->organization_id;
            $assignedUserId = $incident->assigned_to_id && $incident->assigned_to_type === 'user' 
                ? $incident->assigned_to_id 
                : null;

            // 1. Create "created" notification (for organization members)
            // This would have been created when incident was first reported
            if (rand(1, 10) <= 8) { // 80% chance
                $reporter = $agents->random();
                AuditLog::create([
                    'organization_id' => $organizationId,
                    'user_id' => $reporter->id,
                    'action' => 'created',
                    'entity_type' => 'App\Models\Incident',
                    'entity_id' => $incident->id,
                    'before' => null,
                    'after' => [
                        'id' => $incident->id,
                        'title' => $incident->title,
                        'status' => 'REPORTED',
                    ],
                    'ip_address' => $this->randomIpAddress(),
                    'user_agent' => $this->randomUserAgent(),
                    'created_at' => $incident->created_at,
                    'updated_at' => $incident->created_at,
                ]);
                $notificationCount++;
            }

            // 2. Create "assigned" notification (for assigned user)
            if ($assignedUserId && $incident->assigned_at) {
                $assigner = $agents->where('id', '!=', $assignedUserId)->random();
                AuditLog::create([
                    'organization_id' => $organizationId,
                    'user_id' => $assigner->id,
                    'action' => 'assigned',
                    'entity_type' => 'App\Models\Incident',
                    'entity_id' => $incident->id,
                    'before' => [
                        'assigned_to_id' => null,
                        'status' => 'TRIAGED',
                    ],
                    'after' => [
                        'assigned_to_id' => $assignedUserId,
                        'assigned_to_type' => 'user',
                        'status' => 'ASSIGNED',
                    ],
                    'ip_address' => $this->randomIpAddress(),
                    'user_agent' => $this->randomUserAgent(),
                    'created_at' => $incident->assigned_at,
                    'updated_at' => $incident->assigned_at,
                ]);
                $notificationCount++;
            }

            // 3. Create "status_changed" notifications
            // Get status events to create realistic status change notifications
            $statusEvents = DB::table('incident_status_events')
                ->where('incident_id', $incident->id)
                ->whereNotNull('to_status')
                ->orderBy('created_at')
                ->get();

            if ($statusEvents->isNotEmpty()) {
                $previousStatus = 'REPORTED';
                foreach ($statusEvents as $event) {
                    $user = User::find($event->user_id) ?? $agents->random();
                    AuditLog::create([
                        'organization_id' => $organizationId,
                        'user_id' => $user->id,
                        'action' => 'status_changed',
                        'entity_type' => 'App\Models\Incident',
                        'entity_id' => $incident->id,
                    'before' => [
                        'status' => $previousStatus,
                    ],
                    'after' => [
                        'status' => $event->to_status,
                    ],
                        'ip_address' => $this->randomIpAddress(),
                        'user_agent' => $this->randomUserAgent(),
                        'created_at' => $event->created_at,
                        'updated_at' => $event->created_at,
                    ]);
                    $previousStatus = $event->to_status;
                    $notificationCount++;
                }
            } else {
                // If no status events, create some based on current status
                $statusTransitions = [
                    'TRIAGED' => ['from' => 'REPORTED', 'to' => 'TRIAGED'],
                    'ASSIGNED' => ['from' => 'TRIAGED', 'to' => 'ASSIGNED'],
                    'IN_PROGRESS' => ['from' => 'ASSIGNED', 'to' => 'IN_PROGRESS'],
                    'RESOLVED' => ['from' => 'IN_PROGRESS', 'to' => 'RESOLVED'],
                ];

                if (isset($statusTransitions[$incident->status])) {
                    $transition = $statusTransitions[$incident->status];
                    $user = $assignedUserId ? User::find($assignedUserId) : $agents->random();
                    if (!$user) $user = $agents->random();

                    AuditLog::create([
                        'organization_id' => $organizationId,
                        'user_id' => $user->id,
                        'action' => 'status_changed',
                        'entity_type' => 'App\Models\Incident',
                        'entity_id' => $incident->id,
                        'before' => [
                            'status' => $transition['from'],
                        ],
                        'after' => [
                            'status' => $transition['to'],
                        ],
                        'ip_address' => $this->randomIpAddress(),
                        'user_agent' => $this->randomUserAgent(),
                        'created_at' => $incident->updated_at ?? $incident->created_at,
                        'updated_at' => $incident->updated_at ?? $incident->created_at,
                    ]);
                    $notificationCount++;
                }
            }
        }

        $this->command->info("Created {$notificationCount} notification entries in audit_logs.");
    }

    private function randomIpAddress(): string
    {
        return rand(1, 255) . '.' . rand(1, 255) . '.' . rand(1, 255) . '.' . rand(1, 255);
    }

    private function randomUserAgent(): string
    {
        $userAgents = [
            'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
            'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_7) AppleWebKit/537.36',
            'Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36',
            'Mozilla/5.0 (iPhone; CPU iPhone OS 14_0 like Mac OS X) AppleWebKit/605.1.15',
            'Mozilla/5.0 (Android 11; Mobile; rv:68.0) Gecko/68.0',
        ];
        return $userAgents[array_rand($userAgents)];
    }
}

