<?php

namespace Database\Seeders;

use App\Models\IncidentType;
use App\Models\Organization;
use Illuminate\Database\Seeder;

class IncidentTypeSeeder extends Seeder
{
    public function run(): void
    {
        $this->command->info('Creating incident types for organizations...');

        // Get all organizations
        $organizations = Organization::all();

        if ($organizations->isEmpty()) {
            $this->command->warn('No organizations found. Please run OrganizationSeeder first.');
            return;
        }

        // Define incident types by service type
        $incidentTypesByService = [
            'electricity' => [
                ['name' => 'Power Outage', 'estimated_resolution_hours' => 4, 'description' => 'Complete loss of electrical power'],
                ['name' => 'Broken Power Line', 'estimated_resolution_hours' => 8, 'description' => 'Damaged or broken electrical line'],
                ['name' => 'Transformer Failure', 'estimated_resolution_hours' => 24, 'description' => 'Transformer malfunction or failure'],
                ['name' => 'Meter Issue', 'estimated_resolution_hours' => 2, 'description' => 'Problems with electricity meter'],
                ['name' => 'Street Light Out', 'estimated_resolution_hours' => 6, 'description' => 'Non-functional street lighting'],
                ['name' => 'Power Surge', 'estimated_resolution_hours' => 3, 'description' => 'Electrical power surge'],
                ['name' => 'Electrical Fire', 'estimated_resolution_hours' => 12, 'description' => 'Fire caused by electrical issues'],
                ['name' => 'Fallen Utility Pole', 'estimated_resolution_hours' => 16, 'description' => 'Utility pole has fallen or is damaged'],
                ['name' => 'Overhead Wire Damage', 'estimated_resolution_hours' => 10, 'description' => 'Damaged overhead electrical wires'],
                ['name' => 'Substation Issue', 'estimated_resolution_hours' => 48, 'description' => 'Problems at electrical substation'],
            ],
            'water' => [
                ['name' => 'Water Leak', 'estimated_resolution_hours' => 6, 'description' => 'Water leaking from pipes or fixtures'],
                ['name' => 'No Water Supply', 'estimated_resolution_hours' => 12, 'description' => 'Complete loss of water supply'],
                ['name' => 'Water Quality Issue', 'estimated_resolution_hours' => 24, 'description' => 'Poor water quality or contamination'],
                ['name' => 'Broken Pipe', 'estimated_resolution_hours' => 8, 'description' => 'Damaged or broken water pipe'],
                ['name' => 'Meter Problem', 'estimated_resolution_hours' => 2, 'description' => 'Issues with water meter'],
                ['name' => 'Low Water Pressure', 'estimated_resolution_hours' => 4, 'description' => 'Insufficient water pressure'],
                ['name' => 'Water Contamination', 'estimated_resolution_hours' => 36, 'description' => 'Water contamination detected'],
                ['name' => 'Burst Main Pipe', 'estimated_resolution_hours' => 18, 'description' => 'Main water pipe has burst'],
                ['name' => 'Sewer Backup', 'estimated_resolution_hours' => 12, 'description' => 'Sewer system backup'],
                ['name' => 'Pump Station Failure', 'estimated_resolution_hours' => 48, 'description' => 'Water pump station malfunction'],
            ],
            'roads' => [
                ['name' => 'Pothole', 'estimated_resolution_hours' => 8, 'description' => 'Pothole in road surface'],
                ['name' => 'Road Damage', 'estimated_resolution_hours' => 24, 'description' => 'General road surface damage'],
                ['name' => 'Cracked Pavement', 'estimated_resolution_hours' => 12, 'description' => 'Cracks in road pavement'],
                ['name' => 'Sinkhole', 'estimated_resolution_hours' => 48, 'description' => 'Sinkhole in road'],
                ['name' => 'Missing Road Sign', 'estimated_resolution_hours' => 4, 'description' => 'Missing or damaged road sign'],
                ['name' => 'Blocked Drain', 'estimated_resolution_hours' => 6, 'description' => 'Blocked road drainage'],
                ['name' => 'Road Flooding', 'estimated_resolution_hours' => 12, 'description' => 'Road flooded with water'],
                ['name' => 'Debris on Road', 'estimated_resolution_hours' => 2, 'description' => 'Debris blocking road'],
                ['name' => 'Damaged Guardrail', 'estimated_resolution_hours' => 8, 'description' => 'Damaged or missing guardrail'],
                ['name' => 'Uneven Road Surface', 'estimated_resolution_hours' => 16, 'description' => 'Uneven or bumpy road surface'],
            ],
            'waste' => [
                ['name' => 'Overflowing Garbage Bin', 'estimated_resolution_hours' => 4, 'description' => 'Garbage bin is overflowing'],
                ['name' => 'Missed Collection', 'estimated_resolution_hours' => 8, 'description' => 'Waste collection was missed'],
                ['name' => 'Illegal Dumping', 'estimated_resolution_hours' => 12, 'description' => 'Illegal waste dumping site'],
                ['name' => 'Broken Garbage Bin', 'estimated_resolution_hours' => 6, 'description' => 'Damaged or broken garbage container'],
                ['name' => 'Hazardous Waste', 'estimated_resolution_hours' => 24, 'description' => 'Hazardous waste found'],
                ['name' => 'Recycling Issue', 'estimated_resolution_hours' => 8, 'description' => 'Problems with recycling collection'],
                ['name' => 'Waste Odor', 'estimated_resolution_hours' => 4, 'description' => 'Strong odor from waste'],
                ['name' => 'Animal Access to Waste', 'estimated_resolution_hours' => 6, 'description' => 'Animals accessing waste containers'],
                ['name' => 'Waste Spillage', 'estimated_resolution_hours' => 2, 'description' => 'Waste spilled on ground'],
                ['name' => 'Collection Schedule Issue', 'estimated_resolution_hours' => 4, 'description' => 'Issues with waste collection schedule'],
            ],
            'other' => [
                ['name' => 'General Complaint', 'estimated_resolution_hours' => 24, 'description' => 'General service complaint'],
                ['name' => 'Infrastructure Damage', 'estimated_resolution_hours' => 48, 'description' => 'General infrastructure damage'],
                ['name' => 'Public Safety Concern', 'estimated_resolution_hours' => 12, 'description' => 'Public safety issue'],
                ['name' => 'Environmental Issue', 'estimated_resolution_hours' => 36, 'description' => 'Environmental concern'],
                ['name' => 'Noise Complaint', 'estimated_resolution_hours' => 8, 'description' => 'Noise disturbance'],
                ['name' => 'Vandalism', 'estimated_resolution_hours' => 16, 'description' => 'Vandalism or property damage'],
                ['name' => 'Graffiti', 'estimated_resolution_hours' => 6, 'description' => 'Unwanted graffiti'],
                ['name' => 'Public Facility Issue', 'estimated_resolution_hours' => 12, 'description' => 'Problems with public facility'],
                ['name' => 'Traffic Issue', 'estimated_resolution_hours' => 4, 'description' => 'Traffic-related problem'],
                ['name' => 'Emergency Response', 'estimated_resolution_hours' => 1, 'description' => 'Emergency situation requiring immediate response'],
            ],
        ];

        $totalCreated = 0;

        // For each organization, create incident types based on their service areas or default service types
        foreach ($organizations as $organization) {
            // Get service types from organization's service areas, or use default
            $serviceTypes = $organization->serviceAreas()
                ->where('is_active', true)
                ->distinct()
                ->pluck('service_type')
                ->filter()
                ->toArray();

            // If no service areas, assign default service types based on organization name/slug
            if (empty($serviceTypes)) {
                if (str_contains(strtolower($organization->slug), 'electricity') || 
                    str_contains(strtolower($organization->slug), 'cie')) {
                    $serviceTypes = ['electricity'];
                } elseif (str_contains(strtolower($organization->slug), 'water') || 
                          str_contains(strtolower($organization->slug), 'sodeci')) {
                    $serviceTypes = ['water'];
                } else {
                    // Default: assign all service types
                    $serviceTypes = ['electricity', 'water', 'roads', 'waste', 'other'];
                }
            }

            $this->command->info("Creating incident types for organization: {$organization->name} (Service types: " . implode(', ', $serviceTypes) . ")");

            // Create incident types for each service type
            foreach ($serviceTypes as $serviceType) {
                if (!isset($incidentTypesByService[$serviceType])) {
                    $this->command->warn("Unknown service type: {$serviceType} for organization {$organization->name}");
                    continue;
                }

                foreach ($incidentTypesByService[$serviceType] as $typeData) {
                    // Check if incident type already exists for this organization
                    $existing = IncidentType::where('organization_id', $organization->id)
                        ->where('service_type', $serviceType)
                        ->where('name', $typeData['name'])
                        ->first();

                    if (!$existing) {
                        IncidentType::create([
                            'organization_id' => $organization->id,
                            'name' => $typeData['name'],
                            'service_type' => $serviceType,
                            'description' => $typeData['description'] ?? null,
                            'estimated_resolution_hours' => $typeData['estimated_resolution_hours'],
                            'is_active' => true,
                        ]);
                        $totalCreated++;
                    }
                }
            }
        }

        $this->command->info("Created {$totalCreated} incident types across all organizations.");
        
        // Display summary by service type
        $this->command->info("\nService Types Summary:");
        foreach (['electricity', 'water', 'roads', 'waste', 'other'] as $serviceType) {
            $count = IncidentType::where('service_type', $serviceType)
                ->where('is_active', true)
                ->count();
            $this->command->info("  - {$serviceType}: {$count} incident types");
        }
    }
}

