<?php

namespace Database\Seeders;

use App\Models\Incident;
use App\Models\IncidentMedia;
use App\Models\IncidentType;
use App\Models\Organization;
use App\Models\Team;
use App\Models\User;
use Illuminate\Database\Seeder;

class IncidentSeeder extends Seeder
{
    public function run(): void
    {
        $cie = Organization::where('slug', 'cie-electricity-abidjan')->first();
        $sodeci = Organization::where('slug', 'sodeci-water-abidjan')->first();

        $statuses = ['REPORTED', 'TRIAGED', 'ASSIGNED', 'IN_PROGRESS', 'RESOLVED', 'CLOSED'];
        $priorities = ['LOW', 'MEDIUM', 'HIGH', 'CRITICAL'];
        $verificationStates = ['UNVERIFIED', 'VERIFIED', 'DUPLICATE'];

        // Abidjan coordinates (within service areas)
        $abidjanLat = [5.25, 5.35];
        $abidjanLng = [-4.05, -3.95];

        // French incident titles
        $titles = [
            'electricity' => [
                'Panne de courant dans le quartier',
                'Ligne électrique cassée dans la rue',
                'Dysfonctionnement du transformateur',
                'Éclairage public non fonctionnel',
                'Problème de compteur électrique',
                'Surtension électrique signalée',
                'Ligne électrique tombée',
                'Panne d\'électricité prolongée',
                'Court-circuit dans le secteur',
                'Poteau électrique endommagé',
            ],
            'water' => [
                'Fuite d\'eau sur l\'artère principale',
                'Absence d\'approvisionnement en eau dans la zone',
                'Préoccupations concernant la qualité de l\'eau',
                'Canalisation d\'eau cassée',
                'Problème de compteur d\'eau',
                'Basse pression d\'eau',
                'Rapport de contamination de l\'eau',
                'Rupture de canalisation principale',
                'Eau trouble ou colorée',
                'Station de pompage en panne',
            ],
        ];

        // Generate 200 electricity incidents
        if ($cie) {
            $cieIncidentTypes = IncidentType::where('organization_id', $cie->id)->get();
            $cieCitizens = User::where('organization_id', $cie->id)
                ->whereHas('roles', fn($q) => $q->where('name', 'CITIZEN'))
                ->get();
            $cieAgents = User::where('organization_id', $cie->id)
                ->whereHas('roles', fn($q) => $q->where('name', 'FIELD_AGENT'))
                ->get();
            $cieTeam = Team::where('organization_id', $cie->id)->first();

            for ($i = 0; $i < 200; $i++) {
                $status = $statuses[array_rand($statuses)];
                $priority = $priorities[array_rand($priorities)];
                $verification = $verificationStates[array_rand($verificationStates)];

                // Ensure all required fields have values
                $lat = fake()->randomFloat(8, $abidjanLat[0], $abidjanLat[1]);
                $lng = fake()->randomFloat(8, $abidjanLng[0], $abidjanLng[1]);
                $address = fake('fr_FR')->address();
                $description = trim(fake('fr_FR')->paragraph(3) . ' ' . fake('fr_FR')->sentence());
                
                // Ensure latitude and longitude are never 0 or null
                if ($lat == 0 || $lat === null) {
                    $lat = fake()->randomFloat(8, $abidjanLat[0], $abidjanLat[1]);
                }
                if ($lng == 0 || $lng === null) {
                    $lng = fake()->randomFloat(8, $abidjanLng[0], $abidjanLng[1]);
                }
                
                // Ensure description is never empty
                if (empty($description)) {
                    $description = 'Aucune description fournie.';
                }
                
                $incident = Incident::create([
                    'organization_id' => $cie->id,
                    'reporter_user_id' => $cieCitizens->isNotEmpty() ? $cieCitizens->random()->id : null,
                    'reporter_contact' => '+225 07' . rand(10, 99) . ' ' . rand(100000, 999999),
                    'service_type' => 'electricity',
                    'incident_type_id' => $cieIncidentTypes->isNotEmpty() ? $cieIncidentTypes->random()->id : null,
                    'title' => $titles['electricity'][array_rand($titles['electricity'])] . ' - ' . fake('fr_FR')->streetName(),
                    'description' => $description,
                    'latitude' => $lat,
                    'longitude' => $lng,
                    'address_text' => $address ?: 'Adresse non spécifiée',
                    'status' => $status ?: 'REPORTED',
                    'priority' => $priority ?: 'MEDIUM',
                    'verification_state' => $verification,
                    'assigned_to_type' => in_array($status, ['ASSIGNED', 'IN_PROGRESS', 'RESOLVED']) 
                        ? (rand(0, 1) ? 'user' : 'team')
                        : null,
                    'assigned_to_id' => in_array($status, ['ASSIGNED', 'IN_PROGRESS', 'RESOLVED'])
                        ? ($status === 'ASSIGNED' && rand(0, 1) && $cieTeam 
                            ? $cieTeam->id 
                            : ($cieAgents->isNotEmpty() ? $cieAgents->random()->id : null))
                        : null,
                    'resolved_at' => in_array($status, ['RESOLVED', 'CLOSED']) ? now()->subDays(rand(1, 30)) : null,
                    'closed_at' => $status === 'CLOSED' ? now()->subDays(rand(1, 20)) : null,
                ]);

                // Create realistic status timeline
                $timeline = ['REPORTED'];
                if ($incident->status !== 'REPORTED') {
                    $timeline[] = 'TRIAGED';
                    if (in_array($incident->status, ['ASSIGNED', 'IN_PROGRESS', 'RESOLVED', 'CLOSED'])) {
                        $timeline[] = 'ASSIGNED';
                        if (in_array($incident->status, ['IN_PROGRESS', 'RESOLVED', 'CLOSED'])) {
                            $timeline[] = 'IN_PROGRESS';
                            if (in_array($incident->status, ['RESOLVED', 'CLOSED'])) {
                                $timeline[] = 'RESOLVED';
                                if ($incident->status === 'CLOSED') {
                                    $timeline[] = 'CLOSED';
                                }
                            }
                        }
                    }
                }

                // Create status events for each transition
                $assignedAt = null;
                for ($j = 1; $j < count($timeline); $j++) {
                    $createdAt = now()->subDays(rand(1, 60))->subHours(rand(0, 23));
                    $toStatus = $timeline[$j];
                    
                    // If transitioning to ASSIGNED, create assignment record
                    if ($toStatus === 'ASSIGNED' && $incident->assigned_to_id && $incident->assigned_to_type) {
                        $assignedAt = $createdAt;
                        $assignableType = $incident->assigned_to_type === 'user' ? User::class : Team::class;
                        
                        // Get a dispatcher or admin to be the assigner
                        $assigner = User::where('organization_id', $cie->id)
                            ->whereHas('roles', fn($q) => $q->whereIn('name', ['DISPATCHER', 'ORG_ADMIN']))
                            ->first();
                        
                        if ($assigner) {
                            \App\Models\Assignment::create([
                                'incident_id' => $incident->id,
                                'assignable_type' => $assignableType,
                                'assignable_id' => $incident->assigned_to_id,
                                'assigned_by' => $assigner->id,
                                'notes' => fake('fr_FR')->sentence(),
                                'assigned_at' => $assignedAt,
                            ]);
                        }
                    }
                    
                    \App\Models\IncidentStatusEvent::create([
                        'incident_id' => $incident->id,
                        'user_id' => $incident->assigned_to_id ?? null,
                        'from_status' => $timeline[$j - 1],
                        'to_status' => $toStatus,
                        'notes' => fake('fr_FR')->realText(100),
                        'created_at' => $createdAt,
                        'updated_at' => $createdAt,
                    ]);
                }
                
                // Update incident with assigned_at if it was assigned
                if ($assignedAt) {
                    $incident->update(['assigned_at' => $assignedAt]);
                }

                // Add dummy images (2-5 images per incident, 80% chance)
                // Using picsum.photos for working placeholder images
                if (rand(1, 10) <= 8) {
                    $imageCount = rand(2, 5); // Multiple images per incident
                    
                    for ($img = 0; $img < $imageCount; $img++) {
                        $fileName = 'incident-image-' . ($img + 1) . '.jpg';
                        // Use picsum.photos for working placeholder images
                        // Each incident gets unique random images based on incident ID
                        $imageUrl = 'https://picsum.photos/800/600?random=' . ($incident->id * 10 + $img + 1);
                        IncidentMedia::create([
                            'incident_id' => $incident->id,
                            'file_path' => $imageUrl, // Store full URL for external images
                            'file_name' => $fileName,
                            'mime_type' => 'image/jpeg',
                            'file_size' => rand(500000, 2000000), // 500KB - 2MB
                            'media_type' => 'image',
                            'order' => $img,
                        ]);
                    }
                }

                // Add dummy audio files (voice notes, 30% chance)
                if (rand(1, 10) <= 3) {
                    $audioCount = rand(1, 2); // 1-2 audio files per incident
                    
                    for ($aud = 0; $aud < $audioCount; $aud++) {
                        $fileName = 'incident-voice-note-' . ($aud + 1) . '.m4a';
                        IncidentMedia::create([
                            'incident_id' => $incident->id,
                            'file_path' => 'incidents/' . $incident->id . '/' . $fileName,
                            'file_name' => $fileName,
                            'mime_type' => 'audio/mp4', // M4A format
                            'file_size' => rand(100000, 500000), // 100KB - 500KB
                            'media_type' => 'audio',
                            'order' => 100 + $aud, // Place after images
                        ]);
                    }
                }
            }
        }

        // Generate 200 water incidents
        if ($sodeci) {
            $sodeciIncidentTypes = IncidentType::where('organization_id', $sodeci->id)->get();
            $sodeciCitizens = User::where('organization_id', $sodeci->id)
                ->whereHas('roles', fn($q) => $q->where('name', 'CITIZEN'))
                ->get();
            $sodeciAgents = User::where('organization_id', $sodeci->id)
                ->whereHas('roles', fn($q) => $q->where('name', 'FIELD_AGENT'))
                ->get();
            $sodeciTeam = Team::where('organization_id', $sodeci->id)->first();

            for ($i = 0; $i < 200; $i++) {
                $status = $statuses[array_rand($statuses)];
                $priority = $priorities[array_rand($priorities)];
                $verification = $verificationStates[array_rand($verificationStates)];

                // Ensure all required fields have values
                $lat = fake()->randomFloat(8, $abidjanLat[0], $abidjanLat[1]);
                $lng = fake()->randomFloat(8, $abidjanLng[0], $abidjanLng[1]);
                $address = fake('fr_FR')->address();
                $description = trim(fake('fr_FR')->paragraph(3) . ' ' . fake('fr_FR')->sentence());
                
                // Ensure latitude and longitude are never 0 or null
                if ($lat == 0 || $lat === null) {
                    $lat = fake()->randomFloat(8, $abidjanLat[0], $abidjanLat[1]);
                }
                if ($lng == 0 || $lng === null) {
                    $lng = fake()->randomFloat(8, $abidjanLng[0], $abidjanLng[1]);
                }
                
                // Ensure description is never empty
                if (empty($description)) {
                    $description = 'Aucune description fournie.';
                }
                
                $incident = Incident::create([
                    'organization_id' => $sodeci->id,
                    'reporter_user_id' => $sodeciCitizens->isNotEmpty() ? $sodeciCitizens->random()->id : null,
                    'reporter_contact' => '+225 07' . rand(10, 99) . ' ' . rand(100000, 999999),
                    'service_type' => 'water',
                    'incident_type_id' => $sodeciIncidentTypes->isNotEmpty() ? $sodeciIncidentTypes->random()->id : null,
                    'title' => $titles['water'][array_rand($titles['water'])] . ' - ' . fake('fr_FR')->streetName(),
                    'description' => $description,
                    'latitude' => $lat,
                    'longitude' => $lng,
                    'address_text' => $address ?: 'Adresse non spécifiée',
                    'status' => $status ?: 'REPORTED',
                    'priority' => $priority ?: 'MEDIUM',
                    'verification_state' => $verification,
                    'assigned_to_type' => in_array($status, ['ASSIGNED', 'IN_PROGRESS', 'RESOLVED'])
                        ? (rand(0, 1) ? 'user' : 'team')
                        : null,
                    'assigned_to_id' => in_array($status, ['ASSIGNED', 'IN_PROGRESS', 'RESOLVED'])
                        ? ($status === 'ASSIGNED' && rand(0, 1) && $sodeciTeam
                            ? $sodeciTeam->id
                            : ($sodeciAgents->isNotEmpty() ? $sodeciAgents->random()->id : null))
                        : null,
                    'resolved_at' => in_array($status, ['RESOLVED', 'CLOSED']) ? now()->subDays(rand(1, 30)) : null,
                    'closed_at' => $status === 'CLOSED' ? now()->subDays(rand(1, 20)) : null,
                ]);

                // Create realistic status timeline
                $timeline = ['REPORTED'];
                if ($incident->status !== 'REPORTED') {
                    $timeline[] = 'TRIAGED';
                    if (in_array($incident->status, ['ASSIGNED', 'IN_PROGRESS', 'RESOLVED', 'CLOSED'])) {
                        $timeline[] = 'ASSIGNED';
                        if (in_array($incident->status, ['IN_PROGRESS', 'RESOLVED', 'CLOSED'])) {
                            $timeline[] = 'IN_PROGRESS';
                            if (in_array($incident->status, ['RESOLVED', 'CLOSED'])) {
                                $timeline[] = 'RESOLVED';
                                if ($incident->status === 'CLOSED') {
                                    $timeline[] = 'CLOSED';
                                }
                            }
                        }
                    }
                }

                // Create status events for each transition
                $assignedAt = null;
                for ($j = 1; $j < count($timeline); $j++) {
                    $createdAt = now()->subDays(rand(1, 60))->subHours(rand(0, 23));
                    $toStatus = $timeline[$j];
                    
                    // If transitioning to ASSIGNED, create assignment record
                    if ($toStatus === 'ASSIGNED' && $incident->assigned_to_id && $incident->assigned_to_type) {
                        $assignedAt = $createdAt;
                        $assignableType = $incident->assigned_to_type === 'user' ? User::class : Team::class;
                        
                        // Get a dispatcher or admin to be the assigner
                        $assigner = User::where('organization_id', $sodeci->id)
                            ->whereHas('roles', fn($q) => $q->whereIn('name', ['DISPATCHER', 'ORG_ADMIN']))
                            ->first();
                        
                        if ($assigner) {
                            \App\Models\Assignment::create([
                                'incident_id' => $incident->id,
                                'assignable_type' => $assignableType,
                                'assignable_id' => $incident->assigned_to_id,
                                'assigned_by' => $assigner->id,
                                'notes' => fake('fr_FR')->sentence(),
                                'assigned_at' => $assignedAt,
                            ]);
                        }
                    }
                    
                    \App\Models\IncidentStatusEvent::create([
                        'incident_id' => $incident->id,
                        'user_id' => $incident->assigned_to_id ?? null,
                        'from_status' => $timeline[$j - 1],
                        'to_status' => $toStatus,
                        'notes' => fake('fr_FR')->realText(100),
                        'created_at' => $createdAt,
                        'updated_at' => $createdAt,
                    ]);
                }
                
                // Update incident with assigned_at if it was assigned
                if ($assignedAt) {
                    $incident->update(['assigned_at' => $assignedAt]);
                }

                // Add dummy images (2-5 images per incident, 80% chance)
                // Using picsum.photos for working placeholder images
                if (rand(1, 10) <= 8) {
                    $imageCount = rand(2, 5); // Multiple images per incident
                    
                    for ($img = 0; $img < $imageCount; $img++) {
                        $fileName = 'incident-image-' . ($img + 1) . '.jpg';
                        // Use picsum.photos for working placeholder images
                        // Each incident gets unique random images based on incident ID
                        $imageUrl = 'https://picsum.photos/800/600?random=' . ($incident->id * 10 + $img + 1);
                        IncidentMedia::create([
                            'incident_id' => $incident->id,
                            'file_path' => $imageUrl, // Store full URL for external images
                            'file_name' => $fileName,
                            'mime_type' => 'image/jpeg',
                            'file_size' => rand(500000, 2000000), // 500KB - 2MB
                            'media_type' => 'image',
                            'order' => $img,
                        ]);
                    }
                }

                // Add dummy audio files (voice notes, 30% chance)
                if (rand(1, 10) <= 3) {
                    $audioCount = rand(1, 2); // 1-2 audio files per incident
                    
                    for ($aud = 0; $aud < $audioCount; $aud++) {
                        $fileName = 'incident-voice-note-' . ($aud + 1) . '.m4a';
                        IncidentMedia::create([
                            'incident_id' => $incident->id,
                            'file_path' => 'incidents/' . $incident->id . '/' . $fileName,
                            'file_name' => $fileName,
                            'mime_type' => 'audio/mp4', // M4A format
                            'file_size' => rand(100000, 500000), // 100KB - 500KB
                            'media_type' => 'audio',
                            'order' => 100 + $aud, // Place after images
                        ]);
                    }
                }
            }
        }
    }
}

