<?php

namespace Database\Seeders;

use App\Models\Assignment;
use App\Models\Comment;
use App\Models\CommentMedia;
use App\Models\Incident;
use App\Models\User;
use Illuminate\Database\Seeder;

class CommentSeeder extends Seeder
{
    public function run(): void
    {
        // Get some users to comment
        $users = User::whereHas('roles', fn($q) => $q->whereIn('name', ['FIELD_AGENT', 'LEAD_AGENT', 'DISPATCHER']))
            ->get();

        if ($users->isEmpty()) {
            $this->command->info('No users found. Skipping comment seeding.');
            return;
        }

        // Placeholder file paths matching storage structure
        // These represent files that would be stored in storage/app/public/comments/{comment_id}/
        $imageFileNames = [
            'inspection-photo-1.jpg',
            'work-progress-1.jpg',
            'issue-detail-1.jpg',
            'before-repair-1.jpg',
            'after-repair-1.jpg',
            'equipment-photo-1.jpg',
            'location-marker-1.jpg',
            'damage-assessment-1.jpg',
            'completed-work-1.jpg',
            'safety-check-1.jpg',
        ];

        $commentBodies = [
            'On-site inspection completed. Issue confirmed.',
            'Waiting for parts to arrive. Estimated delivery: 2 days.',
            'Work in progress. Will update once complete.',
            'Additional photos attached for reference.',
            'Team dispatched to location. ETA 30 minutes.',
            'Issue resolved. Please verify.',
            'Need additional resources. Escalating to supervisor.',
            'Weather conditions delaying work. Will resume tomorrow.',
            'Customer contacted. Appointment scheduled.',
            'Follow-up required. Customer not available.',
            'Equipment malfunction. Requesting backup team.',
            'Area secured. Proceeding with repairs.',
            'Documentation updated. All photos attached.',
            'Status update: 50% complete.',
            'Final inspection pending.',
        ];

        // Add comments to incidents
        $incidents = Incident::whereIn('status', ['ASSIGNED', 'IN_PROGRESS', 'RESOLVED'])
            ->limit(100)
            ->get();

        $this->command->info("Adding comments to {$incidents->count()} incidents...");

        foreach ($incidents as $incident) {
            // 60% chance of having comments
            if (rand(1, 10) <= 6) {
                $commentCount = rand(1, 5); // 1-5 comments per incident

                for ($i = 0; $i < $commentCount; $i++) {
                    $comment = Comment::create([
                        'commentable_type' => Incident::class,
                        'commentable_id' => $incident->id,
                        'user_id' => $users->random()->id,
                        'body' => $commentBodies[array_rand($commentBodies)],
                        'created_at' => now()->subDays(rand(0, 30))->subHours(rand(0, 23)),
                        'updated_at' => now()->subDays(rand(0, 30))->subHours(rand(0, 23)),
                    ]);

                    // 50% chance of having images in comment
                    if (rand(1, 10) <= 5) {
                        $imageCount = rand(1, 3); // 1-3 images per comment
                        for ($img = 0; $img < $imageCount; $img++) {
                            $fileName = $imageFileNames[array_rand($imageFileNames)];
                            CommentMedia::create([
                                'comment_id' => $comment->id,
                                'file_path' => 'comments/' . $comment->id . '/' . $fileName,
                                'file_name' => $fileName,
                                'mime_type' => 'image/jpeg',
                                'file_size' => rand(500000, 2000000), // 500KB - 2MB
                                'media_type' => 'image',
                                'order' => $img,
                            ]);
                        }
                    }
                }
            }
        }

        // Add comments to assignments
        $assignments = Assignment::whereNotNull('assigned_at')
            ->whereNull('unassigned_at')
            ->limit(50)
            ->get();

        $this->command->info("Adding comments to {$assignments->count()} assignments...");

        foreach ($assignments as $assignment) {
            // 40% chance of having comments
            if (rand(1, 10) <= 4) {
                $commentCount = rand(1, 3); // 1-3 comments per assignment

                for ($i = 0; $i < $commentCount; $i++) {
                    $comment = Comment::create([
                        'commentable_type' => Assignment::class,
                        'commentable_id' => $assignment->id,
                        'user_id' => $users->random()->id,
                        'body' => $commentBodies[array_rand($commentBodies)],
                        'created_at' => now()->subDays(rand(0, 20))->subHours(rand(0, 23)),
                        'updated_at' => now()->subDays(rand(0, 20))->subHours(rand(0, 23)),
                    ]);

                    // 40% chance of having images in comment
                    if (rand(1, 10) <= 4) {
                        $imageCount = rand(1, 2); // 1-2 images per comment
                        for ($img = 0; $img < $imageCount; $img++) {
                            $fileName = $imageFileNames[array_rand($imageFileNames)];
                            CommentMedia::create([
                                'comment_id' => $comment->id,
                                'file_path' => 'comments/' . $comment->id . '/' . $fileName,
                                'file_name' => $fileName,
                                'mime_type' => 'image/jpeg',
                                'file_size' => rand(500000, 2000000),
                                'media_type' => 'image',
                                'order' => $img,
                            ]);
                        }
                    }
                }
            }
        }

        $this->command->info('Comment seeding completed.');
    }
}

