<?php

namespace Database\Seeders;

use App\Models\Incident;
use App\Models\User;
use Illuminate\Database\Seeder;

class AssignIncidentsToAgentSeeder extends Seeder
{
    /**
     * Assign incidents to agent1@cie.ci
     */
    public function run(): void
    {
        $agent = User::where('email', 'agent1@cie.ci')->first();
        
        if (!$agent) {
            $this->command->error('Agent agent1@cie.ci not found. Please run UserSeeder first.');
            return;
        }

        $this->command->info("Assigning incidents to agent: {$agent->name} (ID: {$agent->id})");

        // Get unassigned or REPORTED incidents for CIE organization
        $incidents = Incident::where('organization_id', $agent->organization_id)
            ->where(function ($query) {
                $query->whereNull('assigned_to_id')
                    ->orWhere('status', 'REPORTED')
                    ->orWhere('status', 'TRIAGED');
            })
            ->limit(20) // Assign 20 incidents
            ->get();

        $assignedCount = 0;

        foreach ($incidents as $incident) {
            $incident->update([
                'assigned_to_type' => 'user',
                'assigned_to_id' => $agent->id,
                'status' => 'ASSIGNED',
            ]);

            // Create assignment record
            \App\Models\Assignment::create([
                'incident_id' => $incident->id,
                'assignable_type' => User::class,
                'assignable_id' => $agent->id,
                'assigned_by' => $agent->id,
                'assigned_at' => now(),
            ]);

            // Create status event
            \App\Models\IncidentStatusEvent::create([
                'incident_id' => $incident->id,
                'user_id' => $agent->id,
                'from_status' => $incident->getOriginal('status'),
                'to_status' => 'ASSIGNED',
                'notes' => 'Assigned via seeder',
            ]);

            $assignedCount++;
        }

        $this->command->info("Successfully assigned {$assignedCount} incidents to {$agent->name}");

        // Also assign some IN_PROGRESS incidents
        $inProgressIncidents = Incident::where('organization_id', $agent->organization_id)
            ->where('status', 'ASSIGNED')
            ->where('assigned_to_id', '!=', $agent->id)
            ->limit(5)
            ->get();

        foreach ($inProgressIncidents as $incident) {
            $oldStatus = $incident->status;
            $incident->update([
                'assigned_to_type' => 'user',
                'assigned_to_id' => $agent->id,
                'status' => 'IN_PROGRESS',
            ]);

            \App\Models\IncidentStatusEvent::create([
                'incident_id' => $incident->id,
                'user_id' => $agent->id,
                'from_status' => $oldStatus,
                'to_status' => 'IN_PROGRESS',
                'notes' => 'Reassigned and set to IN_PROGRESS via seeder',
            ]);

            $assignedCount++;
        }

        $this->command->info("Total incidents assigned: {$assignedCount}");
    }
}

