<?php

namespace App\Services;

use App\Models\User;
use App\Models\Incident;
use App\Jobs\SendFcmNotification;
use Illuminate\Support\Facades\Log;

class FcmNotificationService implements NotificationServiceInterface
{
    public function sendToUser(User $user, string $title, string $body, array $data = []): bool
    {
        $tokens = $user->deviceTokens()
            ->where('is_active', true)
            ->pluck('token')
            ->toArray();

        if (empty($tokens)) {
            return false;
        }

        foreach ($tokens as $token) {
            SendFcmNotification::dispatch($token, $title, $body, $data);
        }

        return true;
    }

    public function notifyNewIncident(Incident $incident): void
    {
        $org = $incident->organization;
        
        // Notify dispatchers and org admins
        $users = $org->users()
            ->whereHas('roles', function ($query) {
                $query->whereIn('name', ['DISPATCHER', 'ORG_ADMIN']);
            })
            ->get();

        foreach ($users as $user) {
            $this->sendToUser(
                $user,
                'New Incident Reported',
                "{$incident->title} - {$incident->service_type}",
                [
                    'type' => 'incident.created',
                    'incident_id' => $incident->id,
                ]
            );
        }
    }

    public function notifyIncidentAssigned(Incident $incident, $assignable): void
    {
        if ($assignable instanceof User) {
            $this->sendToUser(
                $assignable,
                'Incident Assigned to You',
                $incident->title,
                [
                    'type' => 'incident.assigned',
                    'incident_id' => $incident->id,
                ]
            );
        } elseif ($assignable instanceof \App\Models\Team) {
            foreach ($assignable->members as $member) {
                $this->sendToUser(
                    $member,
                    'Incident Assigned to Your Team',
                    $incident->title,
                    [
                        'type' => 'incident.assigned',
                        'incident_id' => $incident->id,
                    ]
                );
            }
        }
    }

    public function notifyStatusChange(Incident $incident, string $oldStatus, string $newStatus): void
    {
        // Notify reporter if known
        if ($incident->reporter) {
            $this->sendToUser(
                $incident->reporter,
                'Incident Status Updated',
                "Your incident: {$incident->title} is now {$newStatus}",
                [
                    'type' => 'incident.status_changed',
                    'incident_id' => $incident->id,
                    'status' => $newStatus,
                ]
            );
        }

        // Notify org admins
        $org = $incident->organization;
        $admins = $org->users()
            ->whereHas('roles', function ($query) {
                $query->where('name', 'ORG_ADMIN');
            })
            ->get();

        foreach ($admins as $admin) {
            $this->sendToUser(
                $admin,
                'Incident Status Changed',
                "{$incident->title} - {$oldStatus} → {$newStatus}",
                [
                    'type' => 'incident.status_changed',
                    'incident_id' => $incident->id,
                    'status' => $newStatus,
                ]
            );
        }
    }
}

