<?php

namespace App\Policies;

use App\Models\Incident;
use App\Models\User;

class IncidentPolicy
{
    public function viewAny(User $user): bool
    {
        return true; // All authenticated users can view incidents (scoped by organization and role in controller)
    }

    public function view(User $user, Incident $incident): bool
    {
        // Citizens can only see their own incidents
        if ($user->isCitizen()) {
            return $incident->reporter_user_id === $user->id;
        }

        // Org users can see incidents in their org
        if ($user->organization_id) {
            return $incident->organization_id === $user->organization_id;
        }

        // Super admin can see all
        return $user->isSuperAdmin();
    }

    public function create(User $user): bool
    {
        return true; // Anyone can create incidents (will be routed to org)
    }

    public function update(User $user, Incident $incident): bool
    {
        // Citizens cannot update incidents
        if ($user->isCitizen()) {
            return false;
        }

        // Super admin can update all
        if ($user->isSuperAdmin()) {
            return true;
        }

        // Org users can update incidents in their org
        if ($user->organization_id && $incident->organization_id === $user->organization_id) {
            // Agents can update if assigned to them or their team
            if ($user->isAgent()) {
                if ($incident->assigned_to_type === 'user' && $incident->assigned_to_id === $user->id) {
                    return true;
                }
                if ($incident->assigned_to_type === 'team') {
                    return $user->teams()->where('teams.id', $incident->assigned_to_id)->exists();
                }
                return false;
            }

            // Dispatchers and admins can update any incident in their org
            return $user->isDispatcher() || $user->isOrgAdmin();
        }

        return false;
    }

    public function assign(User $user, Incident $incident): bool
    {
        // Only dispatchers and org admins can assign
        if ($user->isSuperAdmin()) {
            return true;
        }

        if ($user->organization_id && $incident->organization_id === $user->organization_id) {
            return $user->isDispatcher() || $user->isOrgAdmin();
        }

        return false;
    }

    public function unassign(User $user, Incident $incident): bool
    {
        // Only dispatchers and org admins can unassign
        if ($user->isSuperAdmin()) {
            return true;
        }

        if ($user->organization_id && $incident->organization_id === $user->organization_id) {
            return $user->isDispatcher() || $user->isOrgAdmin();
        }

        return false;
    }

    public function addTimelineEvent(User $user, Incident $incident): bool
    {
        // Same permissions as update
        return $this->update($user, $incident);
    }

    public function delete(User $user, Incident $incident): bool
    {
        // Only org admins and super admin can delete
        if ($user->isSuperAdmin()) {
            return true;
        }

        return $user->isOrgAdmin() && $user->organization_id === $incident->organization_id;
    }
}

