<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;
use Illuminate\Database\Eloquent\SoftDeletes;

class User extends Authenticatable
{
    use HasApiTokens, HasFactory, Notifiable, SoftDeletes;

    protected $fillable = [
        'organization_id',
        'name',
        'email',
        'phone',
        'password',
        'is_active',
    ];

    protected $hidden = [
        'password',
        'remember_token',
    ];

    protected $casts = [
        'email_verified_at' => 'datetime',
        'password' => 'hashed',
        'is_active' => 'boolean',
    ];

    public function organization()
    {
        return $this->belongsTo(Organization::class);
    }

    public function roles()
    {
        return $this->belongsToMany(Role::class)->withTimestamps();
    }

    public function hasRole(string $roleName): bool
    {
        return $this->roles()->where('name', $roleName)->exists();
    }

    public function hasAnyRole(array $roleNames): bool
    {
        return $this->roles()->whereIn('name', $roleNames)->exists();
    }

    public function hasPermission(string $permissionName): bool
    {
        return $this->roles()
            ->whereHas('permissions', function ($query) use ($permissionName) {
                $query->where('name', $permissionName);
            })
            ->exists();
    }

    public function incidents()
    {
        return $this->hasMany(Incident::class, 'reporter_user_id');
    }

    public function assignedIncidents()
    {
        return $this->hasMany(Incident::class, 'assigned_to_id')
            ->where('incidents.assigned_to_type', 'user');
    }

    public function teams()
    {
        return $this->belongsToMany(Team::class, 'team_members')->withPivot('role')->withTimestamps();
    }

    public function deviceTokens()
    {
        return $this->hasMany(DeviceToken::class);
    }

    public function isSuperAdmin(): bool
    {
        return $this->hasRole('SUPER_ADMIN');
    }

    public function isOrgAdmin(): bool
    {
        return $this->hasRole('ORG_ADMIN');
    }

    public function isDispatcher(): bool
    {
        return $this->hasRole('DISPATCHER');
    }

    public function isAgent(): bool
    {
        return $this->hasAnyRole(['LEAD_AGENT', 'FIELD_AGENT']);
    }

    public function isLeadAgent(): bool
    {
        return $this->hasRole('LEAD_AGENT');
    }

    public function isFieldAgent(): bool
    {
        return $this->hasRole('FIELD_AGENT');
    }

    public function isCitizen(): bool
    {
        return $this->hasRole('CITIZEN');
    }
}

