<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\MorphTo;

class Incident extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'organization_id',
        'reporter_user_id',
        'reporter_contact',
        'service_type',
        'incident_type_id',
        'title',
        'description',
        'latitude',
        'longitude',
        'address_text',
        'status',
        'priority',
        'verification_state',
        'assigned_to_type',
        'assigned_to_id',
        'triaged_by_id',
        'triaged_at',
        'assigned_by_id',
        'assigned_at',
        'resolved_at',
        'closed_at',
        'resolution_notes',
    ];

    protected $casts = [
        'latitude' => 'decimal:8',
        'longitude' => 'decimal:8',
        'triaged_at' => 'datetime',
        'assigned_at' => 'datetime',
        'resolved_at' => 'datetime',
        'closed_at' => 'datetime',
    ];

    // Status transitions
    const STATUS_TRANSITIONS = [
        'REPORTED' => ['TRIAGED', 'REJECTED'],
        'TRIAGED' => ['ASSIGNED', 'REJECTED'],
        'ASSIGNED' => ['IN_PROGRESS', 'REJECTED'],
        'IN_PROGRESS' => ['RESOLVED', 'REJECTED'],
        'RESOLVED' => ['CLOSED'],
        'CLOSED' => [],
        'REJECTED' => [],
    ];

    public function organization()
    {
        return $this->belongsTo(Organization::class);
    }

    public function reporter()
    {
        return $this->belongsTo(User::class, 'reporter_user_id');
    }

    public function incidentType()
    {
        return $this->belongsTo(IncidentType::class);
    }

    public function media()
    {
        return $this->hasMany(IncidentMedia::class);
    }

    public function statusEvents()
    {
        return $this->hasMany(IncidentStatusEvent::class);
    }

    public function assignments()
    {
        return $this->hasMany(Assignment::class);
    }

    public function comments()
    {
        return $this->morphMany(Comment::class, 'commentable');
    }

    public function assignedToUser()
    {
        return $this->belongsTo(User::class, 'assigned_to_id');
    }

    public function assignedToTeam()
    {
        return $this->belongsTo(Team::class, 'assigned_to_id');
    }

    public function triagedBy()
    {
        return $this->belongsTo(User::class, 'triaged_by_id');
    }

    public function assignedBy()
    {
        return $this->belongsTo(User::class, 'assigned_by_id');
    }

    public function getAssignedToAttribute()
    {
        if (!$this->assigned_to_id) {
            return null;
        }

        if ($this->assigned_to_type === 'user') {
            return User::find($this->assigned_to_id);
        } elseif ($this->assigned_to_type === 'team') {
            return Team::find($this->assigned_to_id);
        }

        return null;
    }

    public function canTransitionTo(string $newStatus): bool
    {
        $allowed = self::STATUS_TRANSITIONS[$this->status] ?? [];
        return in_array($newStatus, $allowed);
    }

    public function transitionTo(string $newStatus, ?User $user = null, ?string $notes = null): bool
    {
        if (!$this->canTransitionTo($newStatus)) {
            return false;
        }

        $oldStatus = $this->status;
        $this->status = $newStatus;

        // Set timestamps
        if ($newStatus === 'RESOLVED') {
            $this->resolved_at = now();
        } elseif ($newStatus === 'CLOSED') {
            $this->closed_at = now();
        }

        $this->save();

        // Create status event
        IncidentStatusEvent::create([
            'incident_id' => $this->id,
            'user_id' => $user?->id,
            'from_status' => $oldStatus,
            'to_status' => $newStatus,
            'notes' => $notes,
            'event_type' => 'STATUS_CHANGE',
            'visibility' => 'PUBLIC_TO_CITIZEN', // Status changes are visible to citizens
        ]);

        return true;
    }
}

