<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Http\Resources\UserResource;
use App\Models\User;
use App\Models\Role;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;

class UserController extends Controller
{
    public function index(Request $request): JsonResponse
    {
        $this->authorize('viewAny', User::class);

        $query = User::query();

        if ($request->user()->isSuperAdmin()) {
            // Super admin can filter by organization_id if provided
            if ($request->has('organization_id')) {
                $query->where('organization_id', $request->organization_id);
            }
            // Otherwise sees all
        } else {
            $query->where('organization_id', $request->user()->organization_id);
        }

        // Filter by role
        if ($request->has('role')) {
            $roleName = $request->role;
            $query->whereHas('roles', function ($q) use ($roleName) {
                $q->where('name', $roleName);
            });
        }

        // Filter by team membership
        if ($request->has('team_id')) {
            $teamId = $request->integer('team_id');
            $query->whereHas('teams', function ($q) use ($teamId) {
                $q->where('teams.id', $teamId);
            });
        }

        $users = $query->with(['roles', 'organization'])
            ->paginate($request->get('per_page', 15));

        return UserResource::collection($users)->response();
    }

    public function store(Request $request): JsonResponse
    {
        $this->authorize('create', User::class);

        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'email', 'unique:users,email'],
            'phone' => ['nullable', 'string'],
            'password' => ['required', 'string', 'min:8'],
            'organization_id' => ['nullable', 'exists:organizations,id'],
            'role_ids' => ['sometimes', 'array'],
            'role_ids.*' => ['exists:roles,id'],
            'role_names' => ['sometimes', 'array'],
            'role_names.*' => ['exists:roles,name'],
        ]);

        $user = User::create([
            'name' => $validated['name'],
            'email' => $validated['email'],
            'phone' => $validated['phone'] ?? null,
            'password' => Hash::make($validated['password']),
            'organization_id' => $validated['organization_id'] ?? $request->user()->organization_id,
        ]);

        // Support both role_ids and role_names
        $roleIds = $validated['role_ids'] ?? [];
        if (isset($validated['role_names']) && !empty($validated['role_names'])) {
            $roleIds = Role::whereIn('name', $validated['role_names'])->pluck('id')->toArray();
        }

        if (empty($roleIds)) {
            return response()->json(['message' => 'At least one role must be provided'], 422);
        }

        $user->roles()->attach($roleIds);

        return response()->json(new UserResource($user->load('roles')), 201);
    }

    public function show(Request $request, User $user): JsonResponse
    {
        $this->authorize('view', $user);

        return response()->json(new UserResource($user->load(['roles', 'organization'])));
    }

    public function update(Request $request, User $user): JsonResponse
    {
        $this->authorize('update', $user);

        $validated = $request->validate([
            'name' => ['sometimes', 'string', 'max:255'],
            'email' => ['sometimes', 'email', 'unique:users,email,' . $user->id],
            'phone' => ['sometimes', 'nullable', 'string'],
            'password' => ['sometimes', 'string', 'min:8'],
            'is_active' => ['sometimes', 'boolean'],
            'role_ids' => ['sometimes', 'array'],
            'role_ids.*' => ['exists:roles,id'],
        ]);

        if (isset($validated['password'])) {
            $validated['password'] = Hash::make($validated['password']);
        }

        $user->update($validated);

        if (isset($validated['role_ids'])) {
            $user->roles()->sync($validated['role_ids']);
        }

        return response()->json(new UserResource($user->load(['roles', 'organization'])));
    }

    public function destroy(User $user): JsonResponse
    {
        $this->authorize('delete', $user);

        $user->delete();

        return response()->json(['message' => 'User deleted successfully']);
    }
}

