<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Models\Team;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class TeamController extends Controller
{
    public function index(Request $request): JsonResponse
    {
        $query = Team::query();

        if ($request->user()->isSuperAdmin()) {
            // Super admin can filter by organization_id if provided
            if ($request->has('organization_id')) {
                $query->where('organization_id', $request->organization_id);
            }
            // Otherwise sees all
        } elseif ($request->user()->organization_id) {
            $query->where('organization_id', $request->user()->organization_id);
        }

        $teams = $query->with('members')->paginate($request->get('per_page', 15));

        return response()->json($teams);
    }

    public function store(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'description' => ['nullable', 'string'],
            'service_type' => ['nullable', 'string'],
            'member_ids' => ['sometimes', 'array'],
            'member_ids.*' => ['exists:users,id'],
        ]);

        $team = Team::create([
            'organization_id' => $request->user()->organization_id,
            'name' => $validated['name'],
            'description' => $validated['description'] ?? null,
            'service_type' => $validated['service_type'] ?? null,
        ]);

        if (isset($validated['member_ids'])) {
            $team->members()->attach($validated['member_ids']);
        }

        return response()->json($team->load('members'), 201);
    }

    public function show(Request $request, Team $team): JsonResponse
    {
        if ($request->user()->organization_id && $team->organization_id !== $request->user()->organization_id) {
            abort(403);
        }

        return response()->json($team->load('members'));
    }

    public function update(Request $request, Team $team): JsonResponse
    {
        if ($request->user()->organization_id && $team->organization_id !== $request->user()->organization_id) {
            abort(403);
        }

        $validated = $request->validate([
            'name' => ['sometimes', 'string', 'max:255'],
            'description' => ['sometimes', 'nullable', 'string'],
            'service_type' => ['sometimes', 'nullable', 'string'],
            'is_active' => ['sometimes', 'boolean'],
        ]);

        $team->update($validated);

        return response()->json($team->load('members'));
    }

    public function updateMembers(Request $request, Team $team): JsonResponse
    {
        if ($request->user()->organization_id && $team->organization_id !== $request->user()->organization_id) {
            abort(403);
        }

        $validated = $request->validate([
            'member_ids' => ['required', 'array'],
            'member_ids.*' => ['exists:users,id'],
        ]);

        $team->members()->sync($validated['member_ids']);

        return response()->json($team->load('members'));
    }

    public function destroy(Team $team): JsonResponse
    {
        if (request()->user()->organization_id && $team->organization_id !== request()->user()->organization_id) {
            abort(403);
        }

        $team->delete();

        return response()->json(['message' => 'Team deleted successfully']);
    }
}

