<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class NotificationController extends Controller
{
    public function index(Request $request): JsonResponse
    {
        $user = $request->user();
        
        // Get notifications from audit logs related to user's incidents
        $notifications = DB::table('audit_logs')
            ->where(function ($query) use ($user) {
                $query->where('user_id', $user->id)
                    ->orWhere(function ($q) use ($user) {
                        // Notifications for incidents assigned to user
                        $q->where('entity_type', 'App\Models\Incident')
                            ->whereIn('entity_id', function ($subQuery) use ($user) {
                                $subQuery->select('id')
                                    ->from('incidents')
                                    ->where('assigned_to_id', $user->id)
                                    ->where('assigned_to_type', 'user');
                            });
                    })
                    ->orWhere(function ($q) use ($user) {
                        // Notifications for incidents in user's organization
                        if ($user->organization_id) {
                            $q->where('entity_type', 'App\Models\Incident')
                                ->whereIn('entity_id', function ($subQuery) use ($user) {
                                    $subQuery->select('id')
                                        ->from('incidents')
                                        ->where('organization_id', $user->organization_id);
                                });
                        }
                    });
            })
            ->whereIn('action', ['created', 'assigned', 'status_changed', 'comment_added'])
            ->orderBy('created_at', 'desc')
            ->limit(50)
            ->get()
            ->map(function ($log) {
                // Merge before and after into expected format
                $before = json_decode($log->before ?? '{}', true);
                $after = json_decode($log->after ?? '{}', true);
                $data = ['old' => $before, 'new' => $after];
                $message = $this->formatNotificationMessage($log->action, $data);
                
                return [
                    'id' => $log->id,
                    'type' => $log->action,
                    'message' => $message,
                    'entity_type' => $log->entity_type,
                    'entity_id' => $log->entity_id,
                    'read' => false, // TODO: Add read status tracking
                    'created_at' => $log->created_at,
                ];
            });

        return response()->json(['notifications' => $notifications]);
    }

    private function formatNotificationMessage(string $action, array $data): string
    {
        switch ($action) {
            case 'created':
                return 'New incident reported';
            case 'assigned':
                return 'Incident assigned to you';
            case 'status_changed':
                $oldStatus = $data['old']['status'] ?? 'UNKNOWN';
                $newStatus = $data['new']['status'] ?? 'UNKNOWN';
                return "Incident status changed from {$oldStatus} to {$newStatus}";
            case 'comment_added':
                $commentBody = $data['new']['comment_body'] ?? 'A comment was added';
                return "New comment: " . $commentBody;
            default:
                return 'Notification';
        }
    }

    public function markAsRead(Request $request, string $id): JsonResponse
    {
        // TODO: Implement read status tracking
        return response()->json(['message' => 'Notification marked as read']);
    }

    public function markAllAsRead(Request $request): JsonResponse
    {
        // TODO: Implement bulk read status tracking
        return response()->json(['message' => 'All notifications marked as read']);
    }
}

