<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Models\IncidentType;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class IncidentTypeController extends Controller
{
    /**
     * Get service types available for the user's organization
     */
    public function getServiceTypes(Request $request): JsonResponse
    {
        $user = $request->user();
        
        if (!$user->organization_id) {
            return response()->json([
                'service_types' => []
            ]);
        }

        // Get distinct service types from incident types for this organization
        $serviceTypes = IncidentType::where('organization_id', $user->organization_id)
            ->where('is_active', true)
            ->distinct()
            ->pluck('service_type')
            ->filter()
            ->values()
            ->toArray();

        // If no incident types exist, return default service types
        if (empty($serviceTypes)) {
            $serviceTypes = ['electricity', 'water', 'roads', 'waste', 'other'];
        }

        return response()->json([
            'service_types' => $serviceTypes
        ]);
    }

    /**
     * Get incident types for the user's organization
     */
    public function index(Request $request): JsonResponse
    {
        $user = $request->user();
        
        $query = IncidentType::query();

        if ($user->isSuperAdmin()) {
            // Super admin can filter by organization_id if provided
            if ($request->has('organization_id')) {
                $query->where('organization_id', $request->organization_id);
            }
            // Otherwise sees all
        } elseif ($user->organization_id) {
            $query->where('organization_id', $user->organization_id);
        }

        $query->where('is_active', true);

        $incidentTypes = $query->orderBy('service_type')
            ->orderBy('name')
            ->get();

        return response()->json([
            'incident_types' => $incidentTypes
        ]);
    }
}

