<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Http\Requests\Incidents\AssignIncidentRequest;
use App\Http\Requests\Incidents\ChangeStatusRequest;
use App\Http\Requests\Incidents\CreateIncidentRequest;
use App\Http\Requests\Incidents\UpdateIncidentRequest;
use App\Http\Resources\IncidentResource;
use App\Models\Incident;
use App\Services\AuditService;
use App\Services\FcmNotificationService;
use App\Services\IncidentRoutingService;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;

class IncidentController extends Controller
{
    public function __construct(
        private IncidentRoutingService $routingService,
        private FcmNotificationService $notificationService,
        private AuditService $auditService
    ) {
        $this->middleware('throttle:' . config('app.rate_limit_incident_creation', 10) . ',1')
            ->only('store');
    }

    public function index(Request $request): JsonResponse
    {
        $user = $request->user();
        $query = Incident::query();

        // Apply tenant isolation and role-based filtering
        if ($user->isCitizen()) {
            $query->where('reporter_user_id', $user->id);
        } elseif ($user->organization_id && !$user->isSuperAdmin()) {
            // Non-super admin users see only their organization
            $query->where('organization_id', $user->organization_id);

            // Role-based filtering
            if ($user->isFieldAgent()) {
                // Field agents see only incidents assigned to them
                $query->where(function ($q) use ($user) {
                    $q->where(function ($q2) use ($user) {
                        $q2->where('assigned_to_type', 'user')
                           ->where('assigned_to_id', $user->id);
                    });
                });
            } elseif ($user->isLeadAgent()) {
                // Lead agents see incidents assigned to them or their teams
                $teamIds = $user->teams()->pluck('teams.id');
                $query->where(function ($q) use ($user, $teamIds) {
                    $q->where(function ($q2) use ($user) {
                        $q2->where('assigned_to_type', 'user')
                           ->where('assigned_to_id', $user->id);
                    })->orWhere(function ($q2) use ($teamIds) {
                        $q2->where('assigned_to_type', 'team')
                           ->whereIn('assigned_to_id', $teamIds);
                    });
                });
            }
            // Dispatchers and org admins see all incidents in org (no additional filter)
        } elseif ($user->isSuperAdmin() && $request->has('organization_id')) {
            // Super admin can filter by organization_id if provided
            $query->where('organization_id', $request->organization_id);
        }
        // Super admin without filter sees all

        // Enhanced filters
        if ($request->has('status')) {
            $statuses = is_array($request->status) ? $request->status : explode(',', $request->status);
            $query->whereIn('status', $statuses);
        }
        if ($request->has('priority')) {
            $query->where('priority', $request->priority);
        }
        if ($request->has('service_type')) {
            $query->where('service_type', $request->service_type);
        }
        if ($request->has('incident_type_id')) {
            $query->where('incident_type_id', $request->incident_type_id);
        }
        if ($request->has('unassigned') && $request->boolean('unassigned')) {
            $query->whereNull('assigned_to_id');
        }
        if ($request->has('mine') && $request->boolean('mine')) {
            $query->where('assigned_to_type', 'user')
                  ->where('assigned_to_id', $user->id);
        }
        if ($request->has('team')) {
            $teamId = $request->integer('team');
            $query->where('assigned_to_type', 'team')
                  ->where('assigned_to_id', $teamId);
        }
        if ($request->has('assigned_to_id')) {
            $query->where('assigned_to_id', $request->assigned_to_id)
                ->where('assigned_to_type', $request->get('assigned_to_type', 'user'));
        }
        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('title', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%");
            });
        }
        if ($request->has('from')) {
            $query->whereDate('created_at', '>=', $request->from);
        }
        if ($request->has('to')) {
            $query->whereDate('created_at', '<=', $request->to);
        }
        // Location bounding box filter
        if ($request->has('min_lat') && $request->has('max_lat') && 
            $request->has('min_lng') && $request->has('max_lng')) {
            $query->whereBetween('latitude', [$request->min_lat, $request->max_lat])
                  ->whereBetween('longitude', [$request->min_lng, $request->max_lng]);
        }

        // Sorting
        $sort = $request->get('sort', 'newest');
        switch ($sort) {
            case 'oldest':
                $query->oldest();
                break;
            case 'priority':
                $query->orderByRaw("FIELD(priority, 'CRITICAL', 'HIGH', 'MEDIUM', 'LOW')")
                      ->latest();
                break;
            case 'aging':
                $query->orderBy('created_at', 'asc');
                break;
            case 'newest':
            default:
                $query->latest();
                break;
        }

        $incidents = $query->with(['media', 'incidentType', 'reporter', 'organization', 'assignedToUser', 'assignedToTeam'])
            ->paginate($request->get('per_page', 15));

        return IncidentResource::collection($incidents)->response();
    }

    public function store(CreateIncidentRequest $request): JsonResponse
    {
        $user = $request->user();

        DB::beginTransaction();
        try {
            // Route to organization
            $organization = $this->routingService->routeToOrganization(
                $request->service_type,
                $request->latitude,
                $request->longitude
            );

            if (!$organization) {
                return response()->json([
                    'message' => 'No organization found for this location and service type.'
                ], 422);
            }

            $incident = Incident::create([
                'organization_id' => $organization->id,
                'reporter_user_id' => $user?->id,
                'reporter_contact' => $request->reporter_contact ?? $user?->phone,
                'service_type' => $request->service_type,
                'incident_type_id' => $request->incident_type_id,
                'title' => $request->title,
                'description' => $request->description,
                'latitude' => $request->latitude,
                'longitude' => $request->longitude,
                'address_text' => $request->address_text,
                'status' => 'REPORTED',
                'priority' => 'MEDIUM',
            ]);

            // Handle media uploads
            if ($request->hasFile('media')) {
                foreach ($request->file('media') as $index => $file) {
                    $path = $file->store('incidents/' . $incident->id, 'public');
                    $incident->media()->create([
                        'file_path' => $path,
                        'file_name' => $file->getClientOriginalName(),
                        'mime_type' => $file->getMimeType(),
                        'file_size' => $file->getSize(),
                        'media_type' => $this->determineMediaType($file->getMimeType()),
                        'order' => $index,
                    ]);
                }
            }

            $this->auditService->log('created', $incident);
            $this->notificationService->notifyNewIncident($incident);

            DB::commit();

            return response()->json(new IncidentResource($incident->load(['media', 'organization'])), 201);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['message' => 'Failed to create incident: ' . $e->getMessage()], 500);
        }
    }

    public function show(Request $request, Incident $incident): JsonResponse
    {
        $this->authorize('view', $incident);

        $incident->load(['media', 'incidentType', 'reporter', 'statusEvents', 'organization']);

        return (new IncidentResource($incident))->response();
    }

    public function update(UpdateIncidentRequest $request, Incident $incident): JsonResponse
    {
        $oldData = $incident->toArray();

        $incident->update($request->validated());

        $this->auditService->log('updated', $incident, $oldData, $incident->toArray());

        return response()->json(new IncidentResource($incident->fresh(['media', 'incidentType'])));
    }

    public function assign(AssignIncidentRequest $request, Incident $incident): JsonResponse
    {
        DB::beginTransaction();
        try {
            $assignableType = $request->assignable_type;
            $assignableId = $request->assignable_id;

            if ($assignableType === 'user') {
                $assignable = \App\Models\User::findOrFail($assignableId);
            } else {
                $assignable = \App\Models\Team::findOrFail($assignableId);
            }

            $oldData = $incident->toArray();

            $incident->update([
                'assigned_to_type' => $assignableType,
                'assigned_to_id' => $assignableId,
                'assigned_by_id' => $request->user()->id,
                'assigned_at' => now(),
                'status' => 'ASSIGNED',
            ]);

            \App\Models\Assignment::create([
                'incident_id' => $incident->id,
                'assignable_type' => $assignableType === 'user' ? \App\Models\User::class : \App\Models\Team::class,
                'assignable_id' => $assignableId,
                'assigned_by' => $request->user()->id,
                'notes' => $request->notes,
                'assigned_at' => now(),
            ]);

            // Create timeline event for assignment
            \App\Models\IncidentStatusEvent::create([
                'incident_id' => $incident->id,
                'user_id' => $request->user()->id,
                'event_type' => 'ASSIGNMENT',
                'to_status' => 'ASSIGNED', // Status after assignment
                'visibility' => 'PUBLIC_TO_CITIZEN',
                'notes' => $request->notes ?? "Assigned to " . ($assignableType === 'user' ? $assignable->name : $assignable->name),
                'metadata' => [
                    'assignable_type' => $assignableType,
                    'assignable_id' => $assignableId,
                    'assigned_by' => $request->user()->id,
                ],
            ]);

            $this->auditService->log('assigned', $incident, $oldData, $incident->toArray());
            $this->notificationService->notifyIncidentAssigned($incident, $assignable);

            DB::commit();

            $incident->refresh();
            $incident->load(['assignedToUser', 'assignedToTeam', 'organization']);
            return (new IncidentResource($incident))->response();
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['message' => 'Failed to assign incident: ' . $e->getMessage()], 500);
        }
    }

    public function unassign(Request $request, Incident $incident): JsonResponse
    {
        $this->authorize('unassign', $incident);

        DB::beginTransaction();
        try {
            $oldData = $incident->toArray();
            $oldAssignee = $incident->assigned_to_id;
            $oldAssigneeType = $incident->assigned_to_type;

            // Mark latest assignment as unassigned
            $latestAssignment = $incident->assignments()
                ->whereNull('unassigned_at')
                ->latest('assigned_at')
                ->first();

            if ($latestAssignment) {
                $latestAssignment->update([
                    'unassigned_at' => now(),
                ]);
            }

            $incident->update([
                'assigned_to_type' => null,
                'assigned_to_id' => null,
            ]);

            // Create timeline event for unassignment
            \App\Models\IncidentStatusEvent::create([
                'incident_id' => $incident->id,
                'user_id' => $request->user()->id,
                'event_type' => 'ASSIGNMENT',
                'to_status' => $incident->status, // Keep current status
                'visibility' => 'PUBLIC_TO_CITIZEN',
                'notes' => $request->input('note', 'Incident unassigned'),
                'metadata' => [
                    'unassigned' => true,
                    'previous_assignable_type' => $oldAssigneeType,
                    'previous_assignable_id' => $oldAssignee,
                ],
            ]);

            $this->auditService->log('unassigned', $incident, $oldData, $incident->toArray());

            DB::commit();

            $incident->refresh();
            $incident->load(['assignedToUser', 'assignedToTeam', 'organization']);
            return (new IncidentResource($incident))->response();
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['message' => 'Failed to unassign incident: ' . $e->getMessage()], 500);
        }
    }

    public function changeStatus(ChangeStatusRequest $request, Incident $incident): JsonResponse
    {
        $oldStatus = $incident->status;
        $newStatus = $request->status;
        $user = $request->user();

        // Track triaged_by when status changes to TRIAGED
        if ($newStatus === 'TRIAGED' && $oldStatus === 'REPORTED') {
            $incident->triaged_by_id = $user->id;
            $incident->triaged_at = now();
        }

        $success = $incident->transitionTo($newStatus, $user, $request->notes);

        if (!$success) {
            return response()->json([
                'message' => "Cannot transition from {$oldStatus} to {$newStatus}"
            ], 422);
        }

        // Reload incident with relationships
        $incident->refresh();
        $incident->load(['assignedToUser', 'assignedToTeam', 'organization', 'triagedBy']);

        $this->auditService->log('status_changed', $incident);
        $this->notificationService->notifyStatusChange($incident, $oldStatus, $newStatus);

        return (new IncidentResource($incident))->response();
    }

    public function timeline(Request $request, Incident $incident): JsonResponse
    {
        $this->authorize('view', $incident);
        $user = $request->user();

        $query = $incident->statusEvents()->with('user');

        // Filter by visibility for citizens
        if ($user->isCitizen()) {
            $query->where('visibility', 'PUBLIC_TO_CITIZEN');
        } elseif ($request->has('visibility')) {
            $query->where('visibility', $request->visibility);
        }

        // Filter by event type
        if ($request->has('type')) {
            $query->where('event_type', $request->type);
        }

        $timeline = $query->orderBy('created_at')
            ->get()
            ->map(function ($event) {
                return [
                    'id' => $event->id,
                    'event_type' => $event->event_type ?? 'STATUS_CHANGE',
                    'from_status' => $event->from_status,
                    'to_status' => $event->to_status,
                    'user' => $event->user ? [
                        'id' => $event->user->id,
                        'name' => $event->user->name,
                    ] : null,
                    'notes' => $event->notes,
                    'visibility' => $event->visibility ?? 'PUBLIC_TO_CITIZEN',
                    'metadata' => $event->metadata ?? [],
                    'created_at' => $event->created_at->toISOString(),
                ];
            });

        return response()->json(['timeline' => $timeline]);
    }

    public function addTimelineEvent(Request $request, Incident $incident): JsonResponse
    {
        $this->authorize('addTimelineEvent', $incident);

        $validated = $request->validate([
            'event_type' => ['required', 'string', 'in:INTERNAL_NOTE,REQUEST_INFO'],
            'notes' => ['required', 'string', 'max:1000'],
            'visibility' => ['sometimes', 'string', 'in:PUBLIC_TO_CITIZEN,INTERNAL_ONLY'],
        ]);

        $event = \App\Models\IncidentStatusEvent::create([
            'incident_id' => $incident->id,
            'user_id' => $request->user()->id,
            'event_type' => $validated['event_type'],
            'to_status' => $incident->status, // Use current status for non-status-change events
            'visibility' => $validated['visibility'] ?? ($validated['event_type'] === 'INTERNAL_NOTE' ? 'INTERNAL_ONLY' : 'PUBLIC_TO_CITIZEN'),
            'notes' => $validated['notes'],
            'metadata' => [],
        ]);

        // If REQUEST_INFO, notify citizen
        if ($validated['event_type'] === 'REQUEST_INFO' && $incident->reporter) {
            $this->notificationService->sendToUser(
                $incident->reporter,
                'More Information Requested',
                "We need more information about your incident: {$incident->title}",
                [
                    'type' => 'incident.info_requested',
                    'incident_id' => $incident->id,
                ]
            );
        }

        $this->auditService->log('timeline_event_added', $incident);

        return response()->json([
            'id' => $event->id,
            'event_type' => $event->event_type,
            'user' => [
                'id' => $event->user->id,
                'name' => $event->user->name,
            ],
            'notes' => $event->notes,
            'visibility' => $event->visibility,
            'created_at' => $event->created_at->toISOString(),
        ], 201);
    }

    /**
     * Add media to an existing incident
     * 
     * POST /api/v1/incidents/{incident}/media
     */
    public function addMedia(Request $request, Incident $incident): JsonResponse
    {
        $this->authorize('update', $incident);

        $request->validate([
            'media' => ['required', 'array', 'max:10'],
            'media.*' => ['file', 'mimes:jpeg,jpg,png,gif,webp,mp4,mov,pdf,m4a,aac,mpeg,mp3', 'max:10240'],
            'visibility' => ['sometimes', 'string', 'in:PUBLIC_TO_CITIZEN,INTERNAL_ONLY'],
            'caption' => ['sometimes', 'nullable', 'string', 'max:500'],
        ]);

        try {
            $uploadedMedia = [];
            $visibility = $request->input('visibility', 'PUBLIC_TO_CITIZEN');
            $caption = $request->input('caption');

            foreach ($request->file('media') as $index => $file) {
                $path = $file->store('incidents/' . $incident->id, 'public');
                $media = $incident->media()->create([
                    'file_path' => $path,
                    'file_name' => $file->getClientOriginalName(),
                    'mime_type' => $file->getMimeType(),
                    'file_size' => $file->getSize(),
                    'media_type' => $this->determineMediaType($file->getMimeType()),
                    'order' => $incident->media()->count() + $index,
                    'visibility' => $visibility,
                    'caption' => $caption,
                ]);
                $uploadedMedia[] = $media;

                // Create timeline event for media upload
                \App\Models\IncidentStatusEvent::create([
                    'incident_id' => $incident->id,
                    'user_id' => $request->user()->id,
                    'event_type' => 'MEDIA_ADDED',
                    'to_status' => $incident->status, // Use current status
                    'visibility' => $visibility,
                    'notes' => $caption ?? 'Media uploaded',
                    'metadata' => [
                        'media_id' => $media->id,
                        'media_type' => $media->media_type,
                    ],
                ]);
            }

            $this->auditService->log('media_added', $incident);

            return response()->json([
                'message' => 'Media uploaded successfully',
                'media' => \App\Http\Resources\IncidentMediaResource::collection(collect($uploadedMedia))
            ], 201);
        } catch (\Exception $e) {
            return response()->json(['message' => 'Failed to upload media: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Determine media type from MIME type
     */
    private function determineMediaType(string $mimeType): string
    {
        if (str_starts_with($mimeType, 'image/')) {
            return 'image';
        } elseif (str_starts_with($mimeType, 'video/')) {
            return 'video';
        } elseif (in_array($mimeType, ['audio/m4a', 'audio/aac', 'audio/mpeg', 'audio/mp3', 'audio/mpeg'])) {
            return 'audio';
        } else {
            return 'document';
        }
    }
}

