<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Models\DeviceToken;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class DeviceTokenController extends Controller
{
    /**
     * Register or update device token for push notifications
     * 
     * POST /api/v1/device-tokens
     * 
     * Body (JSON or form-data):
     * - token: string (required) - FCM device token
     * - device_type: string (required) - "android" or "ios"
     * - device_id: string (optional) - Unique device identifier
     */
    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'token' => ['required', 'string', 'max:255'],
            'device_type' => ['required', 'string', 'in:android,ios,web'],
            'device_id' => ['nullable', 'string', 'max:255'],
        ]);

        if ($validator->fails()) {
            return response()->json([
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        $user = $request->user();

        // Upsert: Update if exists for this user+device, create if new
        $deviceToken = DeviceToken::updateOrCreate(
            [
                'user_id' => $user->id,
                'device_id' => $request->device_id,
                'device_type' => $request->device_type,
            ],
            [
                'token' => $request->token,
                'is_active' => true,
                'last_used_at' => now(),
            ]
        );

        return response()->json([
            'message' => 'Device token registered successfully',
            'device_token' => [
                'id' => $deviceToken->id,
                'device_type' => $deviceToken->device_type,
                'device_id' => $deviceToken->device_id,
                'is_active' => $deviceToken->is_active,
            ]
        ], 201);
    }

    /**
     * Remove device token (e.g., on logout)
     * 
     * DELETE /api/v1/device-tokens/{token}
     */
    public function destroy(Request $request, string $token): JsonResponse
    {
        $user = $request->user();

        $deviceToken = DeviceToken::where('user_id', $user->id)
            ->where('token', $token)
            ->first();

        if (!$deviceToken) {
            return response()->json([
                'message' => 'Device token not found'
            ], 404);
        }

        $deviceToken->delete();

        return response()->json([
            'message' => 'Device token removed successfully'
        ], 200);
    }
}

