<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Http\Requests\Comments\CreateCommentRequest;
use App\Http\Resources\CommentResource;
use App\Models\Assignment;
use App\Models\Comment;
use App\Models\CommentMedia;
use App\Models\Incident;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;

class CommentController extends Controller
{
    public function index(Request $request): JsonResponse
    {
        $request->validate([
            'commentable_type' => ['required', 'string', 'in:incident,assignment'],
            'commentable_id' => ['required', 'integer'],
        ]);

        $commentableType = $request->commentable_type === 'incident' 
            ? Incident::class 
            : Assignment::class;

        $commentable = $commentableType::findOrFail($request->commentable_id);

        // Check authorization
        if ($commentable instanceof Incident) {
            $this->authorize('view', $commentable);
        } else {
            // For assignments, check if user can view the related incident
            $this->authorize('view', $commentable->incident);
        }

        $comments = Comment::where('commentable_type', $commentableType)
            ->where('commentable_id', $commentable->id)
            ->with(['user', 'media'])
            ->latest()
            ->paginate($request->get('per_page', 20));

        return CommentResource::collection($comments)->response();
    }

    public function store(CreateCommentRequest $request): JsonResponse
    {
        $request->validate([
            'commentable_type' => ['required', 'string', 'in:incident,assignment'],
            'commentable_id' => ['required', 'integer'],
        ]);

        $commentableType = $request->commentable_type === 'incident' 
            ? Incident::class 
            : Assignment::class;

        $commentable = $commentableType::findOrFail($request->commentable_id);

        // Check authorization
        if ($commentable instanceof Incident) {
            $this->authorize('view', $commentable);
        } else {
            // For assignments, check if user can view the related incident
            $this->authorize('view', $commentable->incident);
        }

        DB::beginTransaction();
        try {
            $comment = Comment::create([
                'commentable_type' => $commentableType,
                'commentable_id' => $commentable->id,
                'user_id' => $request->user()->id,
                'body' => $request->body,
            ]);

            // Handle media uploads
            if ($request->hasFile('media')) {
                foreach ($request->file('media') as $index => $file) {
                    $path = $file->store('comments/' . $comment->id, 'public');
                    $comment->media()->create([
                        'file_path' => $path,
                        'file_name' => $file->getClientOriginalName(),
                        'mime_type' => $file->getMimeType(),
                        'file_size' => $file->getSize(),
                        'media_type' => str_starts_with($file->getMimeType(), 'image/') ? 'image' : 'video',
                        'order' => $index,
                    ]);
                }
            }

            DB::commit();

            return response()->json(new CommentResource($comment->load('media', 'user')), 201);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['message' => 'Failed to create comment: ' . $e->getMessage()], 500);
        }
    }

    public function destroy(Request $request, Comment $comment): JsonResponse
    {
        // Only the comment author or admin can delete
        if ($comment->user_id !== $request->user()->id && !$request->user()->isSuperAdmin()) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        // Check authorization on the commentable
        $commentable = $comment->commentable;
        if ($commentable instanceof Incident) {
            $this->authorize('view', $commentable);
        } else {
            $this->authorize('view', $commentable->incident);
        }

        // Delete associated media files
        foreach ($comment->media as $media) {
            Storage::disk('public')->delete($media->file_path);
        }

        $comment->delete();

        return response()->json(['message' => 'Comment deleted successfully'], 200);
    }
}

