<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use PDO;
use PDOException;

class CreateDatabase extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'db:create';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Create the database if it does not exist';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $database = config('database.connections.mysql.database');
        $host = config('database.connections.mysql.host');
        $port = config('database.connections.mysql.port');
        $username = config('database.connections.mysql.username');
        $password = config('database.connections.mysql.password');

        if (empty($database)) {
            $this->error('Database name is not set in .env file');
            return 1;
        }

        $this->info("Creating database '{$database}'...");

        try {
            // Try to find MySQL socket on macOS (Homebrew)
            $socket = null;
            $possibleSockets = [
                '/tmp/mysql.sock',
                '/opt/homebrew/var/mysql/mysql.sock',
                '/usr/local/var/mysql/mysql.sock',
            ];
            
            foreach ($possibleSockets as $possibleSocket) {
                if (file_exists($possibleSocket)) {
                    $socket = $possibleSocket;
                    break;
                }
            }

            // Connect to MySQL without specifying a database
            // Use socket if available (common on macOS), otherwise use TCP
            if ($socket && $host === '127.0.0.1') {
                $dsn = "mysql:unix_socket={$socket};charset=utf8mb4";
            } else {
                $dsn = "mysql:host={$host};port={$port};charset=utf8mb4";
            }
            
            $pdo = new PDO($dsn, $username, $password, [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            ]);

            // Create database if it doesn't exist
            $pdo->exec("CREATE DATABASE IF NOT EXISTS `{$database}` CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");

            $this->info("✅ Database '{$database}' created successfully!");
            return 0;
        } catch (PDOException $e) {
            $this->error("❌ Failed to create database: " . $e->getMessage());
            $this->newLine();
            $this->warn("Please check your MySQL credentials in .env file:");
            $this->line("  DB_HOST={$host}");
            $this->line("  DB_PORT={$port}");
            $this->line("  DB_USERNAME={$username}");
            $this->line("  DB_PASSWORD=" . (empty($password) ? '(empty)' : '***'));
            return 1;
        }
    }
}
