# Recreate Notifications Seeder

## Overview

The `RecreateNotificationsSeeder` deletes all existing notification entries from the `audit_logs` table and recreates them based on the current data in your database. This is useful when you need to regenerate notifications after data changes or to fix notification inconsistencies.

## What It Does

1. **Deletes existing notifications**: Removes all `audit_logs` entries with:
   - `entity_type` = `App\Models\Incident`
   - `action` in `['created', 'assigned', 'status_changed']`

2. **Recreates notifications from database data**:
   - **"created" notifications**: One for each incident in the database
   - **"assigned" notifications**: One for each active assignment in the `assignments` table
   - **"status_changed" notifications**: One for each status event in the `incident_status_events` table

## Data Sources

The seeder uses the following tables to recreate notifications:

- **`incidents`**: For "created" notifications
  - Uses `reporter_user_id` if available, otherwise finds a user from the organization
  - Timestamp matches `incident.created_at`

- **`assignments`**: For "assigned" notifications
  - Only processes active assignments (where `unassigned_at` is NULL)
  - Uses `assigned_by` as the user who performed the assignment
  - Timestamp matches `assignment.assigned_at`

- **`incident_status_events`**: For "status_changed" notifications
  - Processes all events with a `to_status`
  - Tracks previous status per incident to maintain proper status transitions
  - Uses `user_id` from the event, or finds a user from the organization
  - Timestamp matches `event.created_at`

## Usage

### Run the seeder directly:

```bash
cd backend
php artisan db:seed --class=RecreateNotificationsSeeder
```

### Or add it to DatabaseSeeder:

```php
// database/seeders/DatabaseSeeder.php
public function run(): void
{
    // ... other seeders ...
    
    $this->call(RecreateNotificationsSeeder::class);
}
```

## Output

The seeder will display progress information:

```
Deleting existing notifications...
Deleted 1234 notification entries from audit_logs.
Recreating notifications from database data...
Processing 500 incidents for 'created' notifications...
Created 500 'created' notifications.
Processing 150 assignments for 'assigned' notifications...
Created 150 'assigned' notifications.
Processing 800 status events for 'status_changed' notifications...
Created 800 'status_changed' notifications.
✅ Successfully recreated 1450 notification entries in audit_logs.
```

## Important Notes

1. **This will delete ALL existing notifications** - Make sure this is what you want before running
2. **Notifications are based on current database state** - If you've deleted incidents, assignments, or status events, their notifications won't be recreated
3. **User assignment** - If a user cannot be found for an action (e.g., reporter is NULL), that notification will be skipped
4. **Status transitions** - The seeder tracks status transitions per incident to ensure proper "before" and "after" states

## When to Use

- After bulk data imports
- After fixing data inconsistencies
- When notifications are missing or incorrect
- After deleting and recreating incidents
- When testing notification functionality

## Prerequisites

- Incidents must exist in the `incidents` table
- Users must exist in the `users` table (for assigning actions)
- Assignments should exist in the `assignments` table (for "assigned" notifications)
- Status events should exist in the `incident_status_events` table (for "status_changed" notifications)

## Example Scenarios

### Scenario 1: After importing incidents
```bash
# Import incidents from external source
php artisan import:incidents

# Recreate notifications for all imported incidents
php artisan db:seed --class=RecreateNotificationsSeeder
```

### Scenario 2: After fixing assignment data
```bash
# Fix assignments in database
php artisan fix:assignments

# Recreate notifications to reflect fixed assignments
php artisan db:seed --class=RecreateNotificationsSeeder
```

### Scenario 3: Clean slate
```bash
# Delete all notifications
php artisan db:seed --class=RecreateNotificationsSeeder
```

