# NE-SOS Backend API

A production-quality multi-tenant incident management SaaS backend built with Laravel 11+ for African utilities and municipalities.

**Powered by [Nation Excellency LLC](https://nationexcellency.com/)**

## Features

- **Multi-tenant Architecture**: Organizations (utilities/municipalities) with complete data isolation
- **Role-Based Access Control (RBAC)**: 6 roles (SUPER_ADMIN, ORG_ADMIN, DISPATCHER, LEAD_AGENT, FIELD_AGENT, CITIZEN)
- **Full Incident Lifecycle**: Report → Triage → Assign → Work → Resolve → Close
- **Geographic Routing**: Service areas route incidents to correct organizations
- **Firebase Notifications**: Push notifications for all key events
- **Audit Logging**: Complete audit trail of all actions
- **RESTful API**: Versioned API (`/api/v1/`) with comprehensive validation

## Tech Stack

- **Laravel 11+** (PHP 8.2+)
- **MySQL 8+**
- **Laravel Sanctum** (SPA token authentication)
- **Firebase Cloud Messaging** (Push notifications)
- **Laravel Queues** (Database driver for VPS simplicity)

## Requirements

- PHP 8.2 or higher
- Composer
- MySQL 8.0 or higher
- Node.js & NPM (for asset compilation, optional)

## Installation

### 1. Clone and Install Dependencies

```bash
composer install
```

### 2. Environment Setup

Copy `.env.example` to `.env`:

```bash
cp .env.example .env
```

Generate application key:

```bash
php artisan key:generate
```

### 3. Configure Environment Variables

Edit `.env` file with your settings:

```env
DB_CONNECTION=mysql
DB_HOST=127.0.0.1
DB_PORT=3306
DB_DATABASE=ne_sos
DB_USERNAME=root
DB_PASSWORD=your_password

# Firebase Configuration
FIREBASE_CREDENTIALS_PATH=/path/to/firebase-credentials.json
FIREBASE_PROJECT_ID=your-project-id
FIREBASE_DATABASE_URL=https://your-project.firebaseio.com

# File Upload Settings
MAX_UPLOAD_SIZE=10240
ALLOWED_MIME_TYPES=image/jpeg,image/png,image/gif,image/webp,video/mp4,video/quicktime,application/pdf

# Rate Limiting
RATE_LIMIT_INCIDENT_CREATION=10
```

### 4. Database Setup

Run migrations:

```bash
php artisan migrate
```

Seed demo data:

```bash
php artisan db:seed
```

This creates:
- 1 SUPER_ADMIN (admin@ne-sos.com / password)
- 2 organizations (CIE Electricity, SODECI Water)
- Service areas for each organization
- Sample users (admins, dispatchers, agents, citizens)
- 100 demo incidents with various statuses

### 5. Storage Link

Create symbolic link for public storage:

```bash
php artisan storage:link
```

### 6. Queue Worker (Required for Notifications)

Start the queue worker:

```bash
php artisan queue:work
```

Or use supervisor for production.

## Docker Setup (Optional)

### Using Docker Compose

```bash
docker-compose up -d
```

This starts:
- MySQL container
- Application container
- Queue worker container

Access the application at `http://localhost:8000`

## API Endpoints

### Authentication

```
POST   /api/auth/login          - Login
POST   /api/auth/logout          - Logout
GET    /api/auth/me              - Get current user
```

### Incidents

```
GET    /api/v1/incidents                    - List incidents (filtered by org)
POST   /api/v1/incidents                    - Create incident
GET    /api/v1/incidents/{id}               - Get incident
PATCH  /api/v1/incidents/{id}               - Update incident
POST   /api/v1/incidents/{id}/assign        - Assign incident
POST   /api/v1/incidents/{id}/status        - Change status
GET    /api/v1/incidents/{id}/timeline      - Get status timeline
```

### Organizations (Super Admin)

```
GET    /api/v1/organizations                - List organizations
POST   /api/v1/organizations                - Create organization
GET    /api/v1/organizations/{id}           - Get organization
PATCH  /api/v1/organizations/{id}           - Update organization
DELETE /api/v1/organizations/{id}           - Delete organization
```

### Users

```
GET    /api/v1/users                        - List users
POST   /api/v1/users                        - Create user
GET    /api/v1/users/{id}                   - Get user
PATCH  /api/v1/users/{id}                   - Update user
DELETE /api/v1/users/{id}                   - Delete user
```

### Teams

```
GET    /api/v1/teams                        - List teams
POST   /api/v1/teams                        - Create team
GET    /api/v1/teams/{id}                   - Get team
PATCH  /api/v1/teams/{id}                   - Update team
PUT    /api/v1/teams/{id}/members           - Update team members
DELETE /api/v1/teams/{id}                   - Delete team
```

## Authentication

All API endpoints (except login) require authentication via Bearer token:

```
Authorization: Bearer {token}
```

Tokens are obtained via `/api/auth/login` and should be included in all subsequent requests.

## Tenant Routing

Incidents are automatically routed to organizations based on:

1. **Service Type**: electricity, water, roads, etc.
2. **Location**: Latitude/longitude coordinates
3. **Service Areas**: Organizations define service areas with bounding boxes

**Routing Logic:**
- System finds service area matching service type and containing the location
- If match found → route to that organization
- If no match → route to default organization for that service type
- If no default → return error

## Roles & Permissions

### SUPER_ADMIN
- Full platform access
- Manage all organizations
- View all incidents across all organizations

### ORG_ADMIN
- Manage users in their organization
- Manage teams and service areas
- Full incident management in their org
- Assign incidents

### DISPATCHER
- Triage incidents (verify, set priority)
- Assign incidents to agents/teams
- Update incident details

### LEAD_AGENT / FIELD_AGENT
- View assigned incidents
- Update assigned incidents
- Change status of assigned incidents

### CITIZEN
- Create incidents
- View only their own incidents
- Receive status updates

## Incident Status Flow

```
REPORTED → TRIAGED → ASSIGNED → IN_PROGRESS → RESOLVED → CLOSED
    ↓         ↓
 REJECTED  REJECTED
```

**Status Transitions:**
- `REPORTED` → `TRIAGED`, `REJECTED`
- `TRIAGED` → `ASSIGNED`, `REJECTED`
- `ASSIGNED` → `IN_PROGRESS`, `REJECTED`
- `IN_PROGRESS` → `RESOLVED`, `REJECTED`
- `RESOLVED` → `CLOSED`
- `CLOSED` → (terminal)
- `REJECTED` → (terminal)

## Notifications

Firebase Cloud Messaging (FCM) notifications are sent for:

- **New Incident**: Notifies dispatchers and org admins
- **Incident Assigned**: Notifies assigned agent/team members
- **Status Change**: Notifies reporter and org admins

Notifications are queued and processed asynchronously.

## Testing

Run tests:

```bash
php artisan test
```

Or with PHPUnit:

```bash
./vendor/bin/phpunit
```

**Test Coverage:**
- Tenant isolation
- RBAC enforcement
- Status transition rules
- Incident routing by service area

## Demo Accounts

After seeding, you can login with:

**Super Admin:**
- Email: `admin@ne-sos.com`
- Password: `password`

**CIE Electricity:**
- Admin: `admin@cie.ci` / `password`
- Dispatcher: `dispatcher@cie.ci` / `password`
- Agent: `agent1@cie.ci` / `password`
- Citizen: `citizen1@cie.ci` / `password`

**SODECI Water:**
- Admin: `admin@sodeci.ci` / `password`
- Dispatcher: `dispatcher@sodeci.ci` / `password`
- Agent: `agent1@sodeci.ci` / `password`
- Citizen: `citizen1@sodeci.ci` / `password`

## File Storage

By default, files are stored in `storage/app/public`. For production, consider:

1. **S3**: Update `FILESYSTEM_DISK` in `.env` to `s3`
2. **CDN**: Configure CDN for public file access
3. **Symlink**: Ensure `storage:link` is run

## Security Features

- **Rate Limiting**: Incident creation endpoint is rate-limited
- **File Validation**: MIME type and size validation
- **Tenant Isolation**: Enforced at query level
- **RBAC**: Policy-based authorization
- **Audit Logging**: All critical actions logged
- **Input Validation**: FormRequest validation on all endpoints

## Production Deployment

### Queue Workers

Use Supervisor to manage queue workers:

```ini
[program:ne-sos-queue]
process_name=%(program_name)s_%(process_num)02d
command=php /path/to/artisan queue:work --sleep=3 --tries=3
autostart=true
autorestart=true
user=www-data
numprocs=2
redirect_stderr=true
stdout_logfile=/path/to/storage/logs/queue.log
```

### Cron Jobs

Add to crontab:

```bash
* * * * * cd /path-to-project && php artisan schedule:run >> /dev/null 2>&1
```

### Environment

Set `APP_ENV=production` and `APP_DEBUG=false` in production.

## Troubleshooting

### Firebase Notifications Not Working

1. Verify `FIREBASE_CREDENTIALS_PATH` points to valid JSON file
2. Check queue worker is running
3. Verify device tokens are registered
4. Check Laravel logs: `storage/logs/laravel.log`

### Tenant Isolation Issues

- Ensure `organization_id` is set on all relevant models
- Check middleware is applied to routes
- Verify policies are correctly implemented

## License

MIT

## Support

For issues and questions, please open an issue on the repository.

