# Notification Seeder Information

## Overview

The `NotificationSeeder` creates audit log entries that generate notifications for users. Notifications are derived from `audit_logs` table entries with specific actions related to incidents.

## How Notifications Work

Notifications are **not stored in a separate table**. Instead, they are generated from `audit_logs` entries by the `NotificationController`:

1. The controller queries `audit_logs` for entries with actions: `created`, `assigned`, `status_changed`
2. Filters by:
   - User's own actions
   - Incidents assigned to the user
   - Incidents in the user's organization
3. Formats the audit log data into notification objects

## Seeder Features

### Notification Types Created

1. **"created" notifications** (80% of incidents)
   - Generated when incidents are first reported
   - Visible to all users in the organization
   - Message: "New incident reported"

2. **"assigned" notifications** (for assigned incidents)
   - Generated when incidents are assigned to users
   - Visible to the assigned user
   - Message: "Incident assigned to you"

3. **"status_changed" notifications** (for all status transitions)
   - Generated from `incident_status_events` table
   - Or created based on current incident status
   - Visible to assigned users and organization members
   - Message: "Incident status changed from {old} to {new}"

### Data Structure

The seeder creates `audit_logs` entries with:

- **organization_id**: Incident's organization
- **user_id**: User who performed the action
- **action**: `created`, `assigned`, or `status_changed`
- **entity_type**: `App\Models\Incident`
- **entity_id**: Incident ID
- **before**: JSON with old state (e.g., `{"status": "REPORTED"}`)
- **after**: JSON with new state (e.g., `{"status": "TRIAGED"}`)
- **ip_address**: Random IP address
- **user_agent**: Random user agent string
- **created_at**: Matches incident/event timestamps

### Notification Visibility

Notifications are visible to:
- **User who performed the action** (`user_id` matches)
- **Assigned user** (if incident is assigned to them)
- **Organization members** (if incident is in their organization)

## Usage

The seeder is automatically called when running:

```bash
php artisan db:seed
```

Or run it individually:

```bash
php artisan db:seed --class=NotificationSeeder
```

## Prerequisites

The seeder requires:
- ✅ Users with roles: `FIELD_AGENT`, `LEAD_AGENT`, or `DISPATCHER`
- ✅ Incidents with status: `ASSIGNED`, `IN_PROGRESS`, `RESOLVED`, or `TRIAGED`
- ✅ `incident_status_events` table (optional, but preferred for realistic data)

These should be created by other seeders that run before `NotificationSeeder`:
- `UserSeeder` - Creates users
- `IncidentSeeder` - Creates incidents
- `AssignIncidentsToAgentSeeder` - Creates assignments

## Controller Fix

**Fixed**: The `NotificationController` was updated to properly use `before` and `after` columns from `audit_logs` instead of a non-existent `changes` column.

**Before**:
```php
$data = json_decode($log->changes ?? '{}', true);
```

**After**:
```php
$before = json_decode($log->before ?? '{}', true);
$after = json_decode($log->after ?? '{}', true);
$data = ['old' => $before, 'new' => $after];
```

## Example Output

```
Creating notifications for 200 incidents...
Created 652 notification entries in audit_logs.
```

## Notification Message Format

- **created**: "New incident reported"
- **assigned**: "Incident assigned to you"
- **status_changed**: "Incident status changed from {OLD_STATUS} to {NEW_STATUS}"

## Database Structure

### audit_logs Table
- `id` - Primary key
- `organization_id` - Organization (nullable)
- `user_id` - User who performed action (nullable)
- `action` - Action type: `created`, `assigned`, `status_changed`
- `entity_type` - Model class: `App\Models\Incident`
- `entity_id` - Incident ID
- `before` - JSON with old state
- `after` - JSON with new state
- `ip_address` - IP address (nullable)
- `user_agent` - User agent string (nullable)
- `created_at` - Timestamp
- `updated_at` - Timestamp

## Notes

- Notifications are generated on-the-fly from audit logs
- No separate notifications table exists
- Read status tracking is TODO (currently always returns `false`)
- Notifications are limited to 50 most recent per user
- Only shows notifications for actions: `created`, `assigned`, `status_changed`

---

**Last Updated**: December 28, 2025

