# Login Debugging Guide

## Issue
Users cannot login with:
- `admin@cie.ci` / `password`
- `dispatcher@cie.ci` / `password`
- `agent1@cie.ci` / `password`

## Steps to Debug

### 1. Verify Users Exist in Database
```bash
cd backend
php artisan tinker
```

Then run:
```php
User::where('email', 'admin@cie.ci')->first();
User::where('email', 'dispatcher@cie.ci')->first();
User::where('email', 'agent1@cie.ci')->first();
```

### 2. Run Seeders if Users Don't Exist
```bash
cd backend
php artisan db:seed --class=UserSeeder
```

Or run all seeders:
```bash
php artisan db:seed
```

### 3. Test Login Endpoint Directly
```bash
curl -X POST http://localhost:8000/api/auth/login \
  -H "Content-Type: application/json" \
  -H "Accept: application/json" \
  -d '{"email":"admin@cie.ci","password":"password"}'
```

### 4. Check CORS Configuration
Verify `config/cors.php` allows requests from frontend origin (usually `http://localhost:5173`)

### 5. Check API URL in Frontend
Verify the frontend is using the correct API URL:
- Check browser console for network errors
- Verify `VITE_API_URL` in `.env` or default `http://localhost:8000/api`

### 6. Check Laravel Logs
```bash
tail -f backend/storage/logs/laravel.log
```

### 7. Verify Password Hashing
If users exist but login fails, check if password needs to be rehashed:
```php
// In tinker
$user = User::where('email', 'admin@cie.ci')->first();
Hash::check('password', $user->password); // Should return true
```

If false, update password:
```php
$user->password = Hash::make('password');
$user->save();
```

## Common Issues

1. **Users don't exist** - Run seeders
2. **Wrong API URL** - Check frontend `.env` or `vite.config.ts`
3. **CORS error** - Check `config/cors.php`
4. **Password mismatch** - Rehash passwords
5. **User inactive** - Check `is_active` field in database


